package server

import (
	"bytes"
	"context"
	"errors"
	"fmt"
	"mime"
	"net"
	"net/http"
	"net/url"
	"strconv"
	"strings"
	"time"

	"gitlab.com/gitlab-org/cluster-integration/gitlab-agent/v16/internal/api"
	"gitlab.com/gitlab-org/cluster-integration/gitlab-agent/v16/internal/gitlab"
	gapi "gitlab.com/gitlab-org/cluster-integration/gitlab-agent/v16/internal/gitlab/api"
	"gitlab.com/gitlab-org/cluster-integration/gitlab-agent/v16/internal/module/event_tracker"
	"gitlab.com/gitlab-org/cluster-integration/gitlab-agent/v16/internal/module/kubernetes_api/rpc"
	"gitlab.com/gitlab-org/cluster-integration/gitlab-agent/v16/internal/module/modserver"
	"gitlab.com/gitlab-org/cluster-integration/gitlab-agent/v16/internal/module/modshared"
	"gitlab.com/gitlab-org/cluster-integration/gitlab-agent/v16/internal/module/usage_metrics"
	"gitlab.com/gitlab-org/cluster-integration/gitlab-agent/v16/internal/tool/cache"
	"gitlab.com/gitlab-org/cluster-integration/gitlab-agent/v16/internal/tool/grpctool"
	"gitlab.com/gitlab-org/cluster-integration/gitlab-agent/v16/internal/tool/httpz"
	"gitlab.com/gitlab-org/cluster-integration/gitlab-agent/v16/internal/tool/logz"
	"gitlab.com/gitlab-org/cluster-integration/gitlab-agent/v16/internal/tool/memz"
	"gitlab.com/gitlab-org/cluster-integration/gitlab-agent/v16/pkg/agentcfg"
	"go.opentelemetry.io/contrib/instrumentation/net/http/otelhttp"
	otelmetric "go.opentelemetry.io/otel/metric"
	"go.opentelemetry.io/otel/propagation"
	"go.opentelemetry.io/otel/trace"
	"go.uber.org/zap"
	"google.golang.org/protobuf/proto"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	"k8s.io/apimachinery/pkg/runtime"
	"k8s.io/apiserver/pkg/endpoints/handlers/negotiation"
)

const (
	readHeaderTimeout = 10 * time.Second
	idleTimeout       = 1 * time.Minute

	gitLabKASCookieName             = "_gitlab_kas"
	authorizationHeaderBearerPrefix = "Bearer " // must end with a space
	tokenSeparator                  = ":"
	tokenTypeCI                     = "ci"
	tokenTypePat                    = "pat"
)

var (
	code2reason = map[int32]metav1.StatusReason{
		// 4xx
		http.StatusBadRequest:            metav1.StatusReasonBadRequest,
		http.StatusUnauthorized:          metav1.StatusReasonUnauthorized,
		http.StatusForbidden:             metav1.StatusReasonForbidden,
		http.StatusNotFound:              metav1.StatusReasonNotFound,
		http.StatusMethodNotAllowed:      metav1.StatusReasonMethodNotAllowed,
		http.StatusNotAcceptable:         metav1.StatusReasonNotAcceptable,
		http.StatusConflict:              metav1.StatusReasonConflict,
		http.StatusGone:                  metav1.StatusReasonGone,
		http.StatusRequestEntityTooLarge: metav1.StatusReasonRequestEntityTooLarge,
		http.StatusUnsupportedMediaType:  metav1.StatusReasonUnsupportedMediaType,
		http.StatusUnprocessableEntity:   metav1.StatusReasonInvalid,
		http.StatusTooManyRequests:       metav1.StatusReasonTooManyRequests,

		// 5xx
		http.StatusInternalServerError: metav1.StatusReasonInternalError,
		http.StatusBadGateway:          metav1.StatusReasonInternalError,
		http.StatusServiceUnavailable:  metav1.StatusReasonServiceUnavailable,
		http.StatusGatewayTimeout:      metav1.StatusReasonTimeout,
	}
)

type proxyUserCacheKey struct {
	agentID    int64
	accessType string
	accessKey  string
	csrfToken  string
}

type kubernetesAPIProxy struct {
	log                      *zap.Logger
	api                      modserver.API
	agentConnPool            func(agentID int64) rpc.KubernetesApiClient
	gitLabClient             gitlab.ClientInterface
	allowedOriginURLs        []string
	allowedAgentsCache       *cache.CacheWithErr[string, *gapi.AllowedAgentsForJob]
	authorizeProxyUserCache  *cache.CacheWithErr[proxyUserCacheKey, *gapi.AuthorizeProxyUserResponse]
	requestCounter           usage_metrics.Counter
	ciAccessRequestCounter   usage_metrics.Counter
	ciAccessAgentsCounter    usage_metrics.UniqueCounter
	ciAccessEventTracker     event_tracker.EventsInterface
	userAccessRequestCounter usage_metrics.Counter
	userAccessAgentsCounter  usage_metrics.UniqueCounter
	userAccessEventTracker   event_tracker.EventsInterface
	patAccessRequestCounter  usage_metrics.Counter
	patAccessAgentsCounter   usage_metrics.UniqueCounter
	patAccessEventTracker    event_tracker.EventsInterface
	responseSerializer       runtime.NegotiatedSerializer
	traceProvider            trace.TracerProvider
	tracePropagator          propagation.TextMapPropagator
	meterProvider            otelmetric.MeterProvider
	serverName               string
	serverVia                string
	// urlPathPrefix is guaranteed to end with / by defaulting.
	urlPathPrefix       string
	listenerGracePeriod time.Duration
	shutdownGracePeriod time.Duration
}

func (p *kubernetesAPIProxy) Run(ctx context.Context, listener net.Listener) error {
	var handler http.Handler
	handler = http.HandlerFunc(p.proxy)
	handler = otelhttp.NewHandler(handler, "k8s-proxy",
		otelhttp.WithTracerProvider(p.traceProvider),
		otelhttp.WithPropagators(p.tracePropagator),
		otelhttp.WithMeterProvider(p.meterProvider),
		otelhttp.WithPublicEndpoint(),
	)
	srv := &http.Server{
		Handler:           handler,
		ReadHeaderTimeout: readHeaderTimeout,
		IdleTimeout:       idleTimeout,
	}
	return httpz.RunServer(ctx, srv, listener, p.listenerGracePeriod, p.shutdownGracePeriod)
}

// proxy Kubernetes API calls via agentk to the cluster Kube API.
//
// This method also takes care of CORS preflight requests as documented [here](https://developer.mozilla.org/en-US/docs/Glossary/Preflight_request).
func (p *kubernetesAPIProxy) proxy(w http.ResponseWriter, r *http.Request) {
	// for preflight and normal requests we want to allow some configured allowed origins and
	// support exposing the response to the client when credentials (e.g. cookies) are included in the request
	header := w.Header()

	requestedOrigin := r.Header.Get(httpz.OriginHeader)
	if requestedOrigin != "" {
		// If the Origin header is set, it needs to match the configured allowed origin urls.
		if !p.isOriginAllowed(requestedOrigin) {
			// Reject the request because origin is not allowed
			p.log.Sugar().Debugf("Received Origin %q is not in configured allowed origins", requestedOrigin)
			w.WriteHeader(http.StatusForbidden)
			return
		}
		header[httpz.AccessControlAllowOriginHeader] = []string{requestedOrigin}
		header[httpz.AccessControlAllowCredentialsHeader] = []string{"true"}
		header[httpz.VaryHeader] = []string{httpz.OriginHeader}
	}
	header[httpz.ServerHeader] = []string{p.serverName} // It will be removed just before responding with actual headers from upstream

	if r.Method == http.MethodOptions {
		// we have a preflight request
		header[httpz.AccessControlAllowHeadersHeader] = r.Header[httpz.AccessControlRequestHeadersHeader]
		// all allowed HTTP methods:
		// see https://developer.mozilla.org/en-US/docs/Web/HTTP/Methods
		header[httpz.AccessControlAllowMethodsHeader] = []string{"GET, HEAD, POST, PUT, DELETE, CONNECT, OPTIONS, TRACE, PATCH"}
		header[httpz.AccessControlMaxAgeHeader] = []string{"86400"}
		w.WriteHeader(http.StatusOK)
	} else {
		log, agentID, eResp := p.proxyInternal(w, r)
		if eResp != nil {
			p.writeErrorResponse(log, agentID)(w, r, eResp)
		}
	}
}

func (p *kubernetesAPIProxy) isOriginAllowed(origin string) bool {
	for _, v := range p.allowedOriginURLs {
		if v == origin {
			return true
		}
	}
	return false
}

func (p *kubernetesAPIProxy) proxyInternal(w http.ResponseWriter, r *http.Request) (*zap.Logger, int64 /* agentID */, *grpctool.ErrResp) {
	ctx := r.Context()
	log := p.log.With(logz.TraceIDFromContext(ctx))

	if !strings.HasPrefix(r.URL.Path, p.urlPathPrefix) {
		msg := "Bad request: URL does not start with expected prefix"
		log.Debug(msg, logz.URLPath(r.URL.Path), logz.URLPathPrefix(p.urlPathPrefix))
		return log, modshared.NoAgentID, &grpctool.ErrResp{
			StatusCode: http.StatusBadRequest,
			Msg:        msg,
		}
	}

	log, agentID, impConfig, creds, eResp := p.authenticateAndImpersonateRequest(ctx, log, r)
	if eResp != nil {
		// If GitLab doesn't authorize the proxy user to make the call,
		// we send an extra header to indicate that, so that the client
		// can differentiate from an *unauthorized* response from GitLab
		// and from an *authorized* response from the proxied K8s cluster.
		if eResp.StatusCode == http.StatusUnauthorized {
			w.Header()[httpz.GitlabUnauthorizedHeader] = []string{"true"}
		}
		return log, agentID, eResp
	}

	p.requestCounter.Inc() // Count only authenticated and authorized requests

	mkClient, err := p.agentConnPool(agentID).MakeRequest(ctx)
	if err != nil {
		msg := "Proxy failed to make outbound request"
		p.api.HandleProcessingError(ctx, log, agentID, msg, err)
		return log, agentID, &grpctool.ErrResp{
			StatusCode: http.StatusInternalServerError,
			Msg:        msg,
			Err:        err,
		}
	}

	p.pipeStreams(log, agentID, w, r, mkClient, impConfig, creds) //nolint: contextcheck
	return log, agentID, nil
}

func (p *kubernetesAPIProxy) authenticateAndImpersonateRequest(ctx context.Context, log *zap.Logger, r *http.Request) (*zap.Logger, int64 /* agentID */ /* userID */, *rpc.ImpersonationConfig, any, *grpctool.ErrResp) {
	agentID, creds, err := getAuthorizationInfoFromRequest(r)
	if err != nil {
		msg := "Unauthorized"
		log.Debug(msg, logz.Error(err))
		return log, modshared.NoAgentID, nil, nil, &grpctool.ErrResp{
			StatusCode: http.StatusUnauthorized,
			Msg:        msg,
			Err:        err,
		}
	}
	log = log.With(logz.AgentID(agentID))
	trace.SpanFromContext(ctx).SetAttributes(api.TraceAgentIDAttr.Int64(agentID))

	var (
		userID    int64
		impConfig *rpc.ImpersonationConfig // can be nil
	)

	switch c := creds.(type) {
	case ciJobTokenAuthn:
		allowedForJob, eResp := p.getAllowedAgentsForJob(ctx, log, agentID, c.jobToken)
		if eResp != nil {
			return log, agentID, nil, creds, eResp
		}
		userID = allowedForJob.User.Id

		aa := findAllowedAgent(agentID, allowedForJob)
		if aa == nil {
			msg := "Forbidden: agentID is not allowed"
			log.Debug(msg)
			return log, agentID, nil, creds, &grpctool.ErrResp{
				StatusCode: http.StatusForbidden,
				Msg:        msg,
			}
		}

		impConfig, err = constructJobImpersonationConfig(allowedForJob, aa)
		if err != nil {
			msg := "Failed to construct impersonation config"
			p.api.HandleProcessingError(ctx, log, agentID, msg, err)
			return log, agentID, nil, creds, &grpctool.ErrResp{
				StatusCode: http.StatusInternalServerError,
				Msg:        msg,
				Err:        err,
			}
		}

		// update usage metrics for `ci_access` requests using the CI tunnel
		p.ciAccessRequestCounter.Inc()
		p.ciAccessAgentsCounter.Add(agentID)
		p.ciAccessEventTracker.EmitEvent(userID, aa.ConfigProject.Id)
	case sessionCookieAuthn:
		auth, eResp := p.authorizeProxyUser(ctx, log, agentID, "session_cookie", c.encryptedPublicSessionID, c.csrfToken)
		if eResp != nil {
			return log, agentID, nil, creds, eResp
		}
		userID = auth.User.Id
		impConfig, err = constructUserImpersonationConfig(auth, "session_cookie")
		if err != nil {
			msg := "Failed to construct user impersonation config"
			p.api.HandleProcessingError(ctx, log, agentID, msg, err)
			return log, agentID, nil, creds, &grpctool.ErrResp{
				StatusCode: http.StatusInternalServerError,
				Msg:        msg,
				Err:        err,
			}
		}

		// update usage metrics for `user_access` requests using the CI tunnel
		p.userAccessRequestCounter.Inc()
		p.userAccessAgentsCounter.Add(agentID)
		p.userAccessEventTracker.EmitEvent(userID, auth.Agent.ConfigProject.Id)
	case patAuthn:
		auth, eResp := p.authorizeProxyUser(ctx, log, agentID, "personal_access_token", c.token, "")
		if eResp != nil {
			return log, agentID, nil, creds, eResp
		}
		userID = auth.User.Id
		impConfig, err = constructUserImpersonationConfig(auth, "personal_access_token")
		if err != nil {
			msg := "Failed to construct user impersonation config"
			p.api.HandleProcessingError(ctx, log, agentID, msg, err)
			return log, agentID, nil, creds, &grpctool.ErrResp{
				StatusCode: http.StatusInternalServerError,
				Msg:        msg,
				Err:        err,
			}
		}

		// update usage metrics for PAT requests using the CI tunnel
		p.patAccessRequestCounter.Inc()
		p.patAccessAgentsCounter.Add(agentID)
		p.patAccessEventTracker.EmitEvent(userID, auth.Agent.ConfigProject.Id)
	default: // This should never happen
		msg := "Invalid authorization type"
		p.api.HandleProcessingError(ctx, log, agentID, msg, err)
		return log, agentID, nil, creds, &grpctool.ErrResp{
			StatusCode: http.StatusInternalServerError,
			Msg:        msg,
		}
	}
	return log, agentID, impConfig, creds, nil
}

func (p *kubernetesAPIProxy) getAllowedAgentsForJob(ctx context.Context, log *zap.Logger, agentID int64, jobToken string) (*gapi.AllowedAgentsForJob, *grpctool.ErrResp) {
	allowedForJob, err := p.allowedAgentsCache.GetItem(ctx, jobToken, func() (*gapi.AllowedAgentsForJob, error) {
		return gapi.GetAllowedAgentsForJob(ctx, p.gitLabClient, jobToken)
	})
	if err != nil {
		var status int32
		var msg string
		switch {
		case gitlab.IsUnauthorized(err):
			status = http.StatusUnauthorized
			msg = "Unauthorized: CI job token"
			log.Debug(msg, logz.Error(err))
		case gitlab.IsForbidden(err):
			status = http.StatusForbidden
			msg = "Forbidden: CI job token"
			log.Debug(msg, logz.Error(err))
		case gitlab.IsNotFound(err):
			status = http.StatusNotFound
			msg = "Not found: agents for CI job token"
			log.Debug(msg, logz.Error(err))
		default:
			status = http.StatusInternalServerError
			msg = "Failed to get allowed agents for CI job token"
			p.api.HandleProcessingError(ctx, log, agentID, msg, err)
		}
		return nil, &grpctool.ErrResp{
			StatusCode: status,
			Msg:        msg,
			Err:        err,
		}
	}
	return allowedForJob, nil
}

func (p *kubernetesAPIProxy) authorizeProxyUser(ctx context.Context, log *zap.Logger, agentID int64, accessType, accessKey, csrfToken string) (*gapi.AuthorizeProxyUserResponse, *grpctool.ErrResp) {
	key := proxyUserCacheKey{
		agentID:    agentID,
		accessType: accessType,
		accessKey:  accessKey,
		csrfToken:  csrfToken,
	}
	auth, err := p.authorizeProxyUserCache.GetItem(ctx, key, func() (*gapi.AuthorizeProxyUserResponse, error) {
		return gapi.AuthorizeProxyUser(ctx, p.gitLabClient, agentID, accessType, accessKey, csrfToken)
	})
	if err != nil {
		switch {
		case gitlab.IsUnauthorized(err), gitlab.IsForbidden(err), gitlab.IsNotFound(err), gitlab.IsBadRequest(err):
			log.Debug("Authorize proxy user error", logz.Error(err))
			return nil, &grpctool.ErrResp{
				StatusCode: http.StatusUnauthorized,
				Msg:        "Unauthorized",
			}
		default:
			msg := "Failed to authorize user session"
			p.api.HandleProcessingError(ctx, log, agentID, msg, err)
			return nil, &grpctool.ErrResp{
				StatusCode: http.StatusInternalServerError,
				Msg:        msg,
			}
		}

	}
	return auth, nil
}

func (p *kubernetesAPIProxy) pipeStreams(log *zap.Logger, agentID int64, w http.ResponseWriter, r *http.Request,
	client rpc.KubernetesApi_MakeRequestClient, impConfig *rpc.ImpersonationConfig, creds any) {

	// urlPathPrefix is guaranteed to end with / by defaulting. That means / will be removed here.
	// Put it back by -1 on length.
	r.URL.Path = r.URL.Path[len(p.urlPathPrefix)-1:]

	// remove GitLab authorization headers (job token, session cookie etc)
	delete(r.Header, httpz.AuthorizationHeader)
	delete(r.Header, httpz.CookieHeader)
	delete(r.Header, httpz.GitlabAgentIDHeader)
	delete(r.Header, httpz.CSRFTokenHeader)
	// remove GitLab authorization query parameters
	query := r.URL.Query()
	delete(query, httpz.GitlabAgentIDQueryParam)
	delete(query, httpz.CSRFTokenQueryParam)
	r.URL.RawQuery = query.Encode()

	r.Header[httpz.ViaHeader] = append(r.Header[httpz.ViaHeader], p.serverVia)

	http2grpc := grpctool.InboundHTTPToOutboundGRPC{
		Log: log,
		HandleProcessingError: func(msg string, err error) {
			p.api.HandleProcessingError(r.Context(), log, agentID, msg, err)
		},
		WriteErrorResponse: p.writeErrorResponse(log, agentID),
		MergeHeaders:       p.mergeProxiedResponseHeaders,
		CheckHeader: func(statusCode int32, header http.Header) error {
			if !isCookieAuthn(creds) {
				// The below checks only apply to requests that authenticate using a cookie.
				return nil
			}
			if isHTTPRedirectStatusCode(statusCode) {
				// Do not proxy redirects for requests with Cookie + CSRF token authentication.
				return errors.New("redirects are not allowed")
			}
			err := checkContentType(header, runtime.ContentTypeJSON, runtime.ContentTypeYAML, runtime.ContentTypeProtobuf)
			if err != nil {
				return err
			}

			if len(header[httpz.SetCookieHeader]) > 0 {
				delete(header, httpz.SetCookieHeader)
				log.Debug("Deleted Set-Cookie from the server's response")
			}
			return nil
		},
	}
	var extra proto.Message // don't use a concrete type here or extra will be passed as a typed nil.
	if impConfig != nil {
		extra = &rpc.HeaderExtra{
			ImpConfig: impConfig,
		}
	}
	http2grpc.Pipe(client, w, r, extra)
}

func (p *kubernetesAPIProxy) mergeProxiedResponseHeaders(outbound, inbound http.Header) {
	delete(inbound, httpz.ServerHeader) // remove the header we've added above. We use Via instead.
	// remove all potential CORS headers from the proxied response
	delete(outbound, httpz.AccessControlAllowOriginHeader)
	delete(outbound, httpz.AccessControlAllowHeadersHeader)
	delete(outbound, httpz.AccessControlAllowCredentialsHeader)
	delete(outbound, httpz.AccessControlAllowMethodsHeader)
	delete(outbound, httpz.AccessControlMaxAgeHeader)

	// set headers from proxied response without overwriting the ones already set (e.g. CORS headers)
	for k, vals := range outbound {
		if len(inbound[k]) == 0 {
			inbound[k] = vals
		}
	}
	// explicitly merge Vary header with the headers from proxies requests.
	// We always set the Vary header to `Origin` for CORS
	if v := append(inbound[httpz.VaryHeader], outbound[httpz.VaryHeader]...); len(v) > 0 {
		inbound[httpz.VaryHeader] = v
	}
	inbound[httpz.ViaHeader] = append(inbound[httpz.ViaHeader], p.serverVia)
}

func (p *kubernetesAPIProxy) writeErrorResponse(log *zap.Logger, agentID int64) grpctool.WriteErrorResponse {
	return func(w http.ResponseWriter, r *http.Request, errResp *grpctool.ErrResp) {
		_, info, err := negotiation.NegotiateOutputMediaType(r, p.responseSerializer, negotiation.DefaultEndpointRestrictions)
		ctx := r.Context()
		if err != nil {
			msg := "Failed to negotiate output media type"
			log.Debug(msg, logz.Error(err))
			http.Error(w, formatStatusMessage(ctx, msg, err), http.StatusNotAcceptable)
			return
		}
		message := formatStatusMessage(ctx, errResp.Msg, errResp.Err)
		s := &metav1.Status{
			TypeMeta: metav1.TypeMeta{
				Kind:       "Status",
				APIVersion: "v1",
			},
			Status:  metav1.StatusFailure,
			Message: message,
			Reason:  code2reason[errResp.StatusCode], // if mapping is not present, then "" means metav1.StatusReasonUnknown
			Code:    errResp.StatusCode,
		}
		buf := memz.Get32k() // use a temporary buffer to segregate I/O errors and encoding errors
		defer memz.Put32k(buf)
		buf = buf[:0] // don't care what's in the buf, start writing from the start
		b := bytes.NewBuffer(buf)
		err = info.Serializer.Encode(s, b) // encoding errors
		if err != nil {
			p.api.HandleProcessingError(ctx, log, agentID, "Failed to encode status response", err)
			http.Error(w, message, int(errResp.StatusCode))
			return
		}
		w.Header()[httpz.ContentTypeHeader] = []string{info.MediaType}
		w.WriteHeader(int(errResp.StatusCode))
		_, _ = w.Write(b.Bytes()) // I/O errors
	}
}

func isHTTPRedirectStatusCode(statusCode int32) bool {
	return statusCode >= 300 && statusCode < 400
}

func isCookieAuthn(creds any) bool {
	_, ok := creds.(sessionCookieAuthn)
	return ok
}

func checkContentType(h http.Header, allowed ...string) error {
	// There should be at most one Content-Type header, but it's not our job to do something about it if there is more.
	// We just ensure thy are all allowed.
nextContentType:
	for _, ct := range h[httpz.ContentTypeHeader] {
		mediatype, _, err := mime.ParseMediaType(ct)
		if err != nil && err != mime.ErrInvalidMediaParameter { //nolint:errorlint
			// Parsing error and not a MIME parameter parsing error, which we ignore.
			return fmt.Errorf("check Content-Type: %w", err)
		}
		for _, a := range allowed {
			if mediatype == a {
				// This one is allowed, onto the next Content-Type header
				continue nextContentType
			}
		}
		return fmt.Errorf("Content-Type not allowed: %s", mediatype)
	}
	return nil
}

// err can be nil.
func formatStatusMessage(ctx context.Context, msg string, err error) string {
	var b strings.Builder
	b.WriteString("GitLab Agent Server: ")
	b.WriteString(msg)
	if err != nil {
		b.WriteString(": ")
		b.WriteString(err.Error())
	}
	traceID := trace.SpanContextFromContext(ctx).TraceID()
	if traceID.IsValid() {
		b.WriteString(". Trace ID: ")
		b.WriteString(traceID.String())
	}
	return b.String()
}

func findAllowedAgent(agentID int64, agentsForJob *gapi.AllowedAgentsForJob) *gapi.AllowedAgent {
	for _, aa := range agentsForJob.AllowedAgents {
		if aa.Id == agentID {
			return aa
		}
	}
	return nil
}

type ciJobTokenAuthn struct {
	jobToken string
}

type patAuthn struct {
	token string
}

type sessionCookieAuthn struct {
	encryptedPublicSessionID string
	csrfToken                string
}

func getAuthorizationInfoFromRequest(r *http.Request) (int64 /* agentID */, any, error) {
	if authzHeader := r.Header[httpz.AuthorizationHeader]; len(authzHeader) >= 1 {
		if len(authzHeader) > 1 {
			return 0, nil, fmt.Errorf("%s header: expecting a single header, got %d", httpz.AuthorizationHeader, len(authzHeader))
		}
		agentID, tokenType, token, err := getAgentIDAndTokenFromHeader(authzHeader[0])
		if err != nil {
			return 0, nil, err
		}
		switch tokenType {
		case tokenTypeCI:
			return agentID, ciJobTokenAuthn{
				jobToken: token,
			}, nil
		case tokenTypePat:
			return agentID, patAuthn{
				token: token,
			}, nil
		}
	}
	if cookie, err := r.Cookie(gitLabKASCookieName); err == nil {
		agentID, encryptedPublicSessionID, csrfToken, err := getSessionCookieParams(cookie, r)
		if err != nil {
			return 0, nil, err
		}
		return agentID, sessionCookieAuthn{
			encryptedPublicSessionID: encryptedPublicSessionID,
			csrfToken:                csrfToken,
		}, nil
	}
	return 0, nil, errors.New("no valid credentials provided")
}

func getSessionCookieParams(cookie *http.Cookie, r *http.Request) (int64, string, string, error) {
	if len(cookie.Value) == 0 {
		return 0, "", "", fmt.Errorf("%s cookie value must not be empty", gitLabKASCookieName)
	}
	// NOTE: GitLab Rails uses `rack` as the generic web server framework, which escapes the cookie values.
	// See https://github.com/rack/rack/blob/0b26518acc4c946ca96dfe3d9e68a05ca84439f7/lib/rack/utils.rb#L300
	encryptedPublicSessionID, err := url.QueryUnescape(cookie.Value)
	if err != nil {
		return 0, "", "", fmt.Errorf("%s invalid cookie value", gitLabKASCookieName)
	}

	agentID, err := getAgentIDForSessionCookieRequest(r)
	if err != nil {
		return 0, "", "", err
	}
	csrfToken, err := getCSRFTokenForSessionCookieRequest(r)
	if err != nil {
		return 0, "", "", err
	}
	return agentID, encryptedPublicSessionID, csrfToken, nil
}

// getAgentIDForSessionCookieRequest retrieves the agent id from the request when trying to authenticate with a session cookie.
// First, the agent id is tried to be retrieved from the headers.
// If that fails, the query parameters are tried.
// When both the agent id is provided in the headers and the query parameters the query parameter
// has precedence and the query parameter is silently ignored.
func getAgentIDForSessionCookieRequest(r *http.Request) (int64, error) {
	parseAgentID := func(agentIDStr string) (int64, error) {
		agentID, err := strconv.ParseInt(agentIDStr, 10, 64)
		if err != nil {
			return 0, fmt.Errorf("agent id in request: invalid value: %q", agentIDStr)
		}
		return agentID, nil
	}
	// Check the agent id header and return if it is present and a valid
	agentIDHeader := r.Header[httpz.GitlabAgentIDHeader]
	if len(agentIDHeader) == 1 {
		return parseAgentID(agentIDHeader[0])
	}

	// If multiple agent id headers are given we abort with a failure
	if len(agentIDHeader) > 1 {
		return 0, fmt.Errorf("%s header must have exactly one value", httpz.GitlabAgentIDHeader)
	}

	// Check the query parameters for a valid agent id
	agentIDParam := r.URL.Query()[httpz.GitlabAgentIDQueryParam]
	if len(agentIDParam) != 1 {
		return 0, fmt.Errorf("exactly one agent id must be provided either in the %q header or %q query parameter", httpz.GitlabAgentIDHeader, httpz.GitlabAgentIDQueryParam)
	}

	return parseAgentID(agentIDParam[0])
}

// getCSRFTokenForSessionCookieRequest retrieves the CSRF token from the request when trying to authenticate with a session cookie.
// First, the CSRF token is tried to be retrieved from the headers.
// If that fails, the query parameters are tried.
// When both the CSRF token is provided in the headers and the query parameters the query parameter
// has precedence and the query parameter is silently ignored.
func getCSRFTokenForSessionCookieRequest(r *http.Request) (string, error) {
	// Check the CSRF token header and return if it is present
	csrfTokenHeader := r.Header[httpz.CSRFTokenHeader]
	if len(csrfTokenHeader) == 1 {
		return csrfTokenHeader[0], nil
	}

	// If multiple CSRF tokens headers are given we abort with a failure
	if len(csrfTokenHeader) > 1 {
		return "", fmt.Errorf("%s header must have exactly one value", httpz.CSRFTokenHeader)
	}

	// Check the query parameters for a valid CSRF token
	csrfTokenParam := r.URL.Query()[httpz.CSRFTokenQueryParam]
	if len(csrfTokenParam) != 1 {
		return "", fmt.Errorf("exactly one CSRF token must be provided either in the %q header or %q query parameter", httpz.CSRFTokenHeader, httpz.CSRFTokenQueryParam)
	}

	return csrfTokenParam[0], nil
}

func getAgentIDAndTokenFromHeader(header string) (int64, string /* token type */, string /* token */, error) {
	if !strings.HasPrefix(header, authorizationHeaderBearerPrefix) {
		// "missing" space in message - it's in the authorizationHeaderBearerPrefix constant already
		return 0, "", "", fmt.Errorf("%s header: expecting %stoken", httpz.AuthorizationHeader, authorizationHeaderBearerPrefix)
	}
	tokenValue := header[len(authorizationHeaderBearerPrefix):]
	tokenType, tokenContents, found := strings.Cut(tokenValue, tokenSeparator)
	if !found {
		return 0, "", "", fmt.Errorf("%s header: invalid value", httpz.AuthorizationHeader)
	}
	switch tokenType {
	case tokenTypeCI:
	case tokenTypePat:
	default:
		return 0, "", "", fmt.Errorf("%s header: unknown token type", httpz.AuthorizationHeader)
	}
	agentIDAndToken := tokenContents
	agentIDStr, token, found := strings.Cut(agentIDAndToken, tokenSeparator)
	if !found {
		return 0, "", "", fmt.Errorf("%s header: invalid value", httpz.AuthorizationHeader)
	}
	agentID, err := strconv.ParseInt(agentIDStr, 10, 64)
	if err != nil {
		return 0, "", "", fmt.Errorf("%s header: failed to parse: %w", httpz.AuthorizationHeader, err)
	}
	if token == "" {
		return 0, "", "", fmt.Errorf("%s header: empty token", httpz.AuthorizationHeader)
	}
	return agentID, tokenType, token, nil
}

func constructJobImpersonationConfig(allowedForJob *gapi.AllowedAgentsForJob, aa *gapi.AllowedAgent) (*rpc.ImpersonationConfig, error) {
	as := aa.GetConfiguration().GetAccessAs().GetAs() // all these fields are optional, so handle nils.
	switch imp := as.(type) {
	case nil, *agentcfg.CiAccessAsCF_Agent: // nil means default value, which is Agent.
		return nil, nil
	case *agentcfg.CiAccessAsCF_Impersonate:
		i := imp.Impersonate
		return &rpc.ImpersonationConfig{
			Username: i.Username,
			Groups:   i.Groups,
			Uid:      i.Uid,
			Extra:    impImpersonationExtra(i.Extra),
		}, nil
	case *agentcfg.CiAccessAsCF_CiJob:
		return &rpc.ImpersonationConfig{
			Username: fmt.Sprintf("gitlab:ci_job:%d", allowedForJob.Job.Id),
			Groups:   impCIJobGroups(allowedForJob),
			Extra:    impCIJobExtra(allowedForJob, aa),
		}, nil
	default:
		// Normally this should never happen
		return nil, fmt.Errorf("unexpected job impersonation mode: %T", imp)
	}
}

func constructUserImpersonationConfig(auth *gapi.AuthorizeProxyUserResponse, accessType string) (*rpc.ImpersonationConfig, error) {
	switch imp := auth.GetAccessAs().AccessAs.(type) {
	case *gapi.AccessAsProxyAuthorization_Agent:
		return nil, nil
	case *gapi.AccessAsProxyAuthorization_User:
		return &rpc.ImpersonationConfig{
			Username: fmt.Sprintf("gitlab:user:%s", auth.User.Username),
			Groups:   impUserGroups(auth),
			Extra:    impUserExtra(auth, accessType),
		}, nil
	default:
		// Normally this should never happen
		return nil, fmt.Errorf("unexpected user impersonation mode: %T", imp)
	}
}

func impImpersonationExtra(in []*agentcfg.ExtraKeyValCF) []*rpc.ExtraKeyVal {
	out := make([]*rpc.ExtraKeyVal, 0, len(in))
	for _, kv := range in {
		out = append(out, &rpc.ExtraKeyVal{
			Key: kv.Key,
			Val: kv.Val,
		})
	}
	return out
}

func impCIJobGroups(allowedForJob *gapi.AllowedAgentsForJob) []string {
	// 1. gitlab:ci_job to identify all requests coming from CI jobs.
	groups := make([]string, 0, 3+len(allowedForJob.Project.Groups))
	groups = append(groups, "gitlab:ci_job")
	// 2. The list of ids of groups the project is in.
	for _, projectGroup := range allowedForJob.Project.Groups {
		groups = append(groups, fmt.Sprintf("gitlab:group:%d", projectGroup.Id))

		// 3. The tier of the environment this job belongs to, if set.
		if allowedForJob.Environment != nil {
			groups = append(groups, fmt.Sprintf("gitlab:group_env_tier:%d:%s", projectGroup.Id, allowedForJob.Environment.Tier))
		}
	}
	// 4. The project id.
	groups = append(groups, fmt.Sprintf("gitlab:project:%d", allowedForJob.Project.Id))
	// 5. The slug and tier of the environment this job belongs to, if set.
	if allowedForJob.Environment != nil {
		groups = append(groups,
			fmt.Sprintf("gitlab:project_env:%d:%s", allowedForJob.Project.Id, allowedForJob.Environment.Slug),
			fmt.Sprintf("gitlab:project_env_tier:%d:%s", allowedForJob.Project.Id, allowedForJob.Environment.Tier),
		)
	}
	return groups
}

func impCIJobExtra(allowedForJob *gapi.AllowedAgentsForJob, aa *gapi.AllowedAgent) []*rpc.ExtraKeyVal {
	extra := []*rpc.ExtraKeyVal{
		{
			Key: "agent.gitlab.com/id",
			Val: []string{strconv.FormatInt(aa.Id, 10)}, // agent id
		},
		{
			Key: "agent.gitlab.com/config_project_id",
			Val: []string{strconv.FormatInt(aa.ConfigProject.Id, 10)}, // agent's configuration project id
		},
		{
			Key: "agent.gitlab.com/project_id",
			Val: []string{strconv.FormatInt(allowedForJob.Project.Id, 10)}, // CI project id
		},
		{
			Key: "agent.gitlab.com/ci_pipeline_id",
			Val: []string{strconv.FormatInt(allowedForJob.Pipeline.Id, 10)}, // CI pipeline id
		},
		{
			Key: "agent.gitlab.com/ci_job_id",
			Val: []string{strconv.FormatInt(allowedForJob.Job.Id, 10)}, // CI job id
		},
		{
			Key: "agent.gitlab.com/username",
			Val: []string{allowedForJob.User.Username}, // username of the user the CI job is running as
		},
	}
	if allowedForJob.Environment != nil {
		extra = append(extra,
			&rpc.ExtraKeyVal{
				Key: "agent.gitlab.com/environment_slug",
				Val: []string{allowedForJob.Environment.Slug}, // slug of the environment, if set
			},
			&rpc.ExtraKeyVal{
				Key: "agent.gitlab.com/environment_tier",
				Val: []string{allowedForJob.Environment.Tier}, // tier of the environment, if set
			},
		)
	}
	return extra
}

func impUserGroups(auth *gapi.AuthorizeProxyUserResponse) []string {
	groups := []string{"gitlab:user"}
	for _, accessCF := range auth.AccessAs.GetUser().Projects {
		for _, role := range accessCF.Roles {
			groups = append(groups, fmt.Sprintf("gitlab:project_role:%d:%s", accessCF.Id, role))
		}
	}
	for _, accessCF := range auth.AccessAs.GetUser().Groups {
		for _, role := range accessCF.Roles {
			groups = append(groups, fmt.Sprintf("gitlab:group_role:%d:%s", accessCF.Id, role))
		}
	}
	return groups
}

func impUserExtra(auth *gapi.AuthorizeProxyUserResponse, accessType string) []*rpc.ExtraKeyVal {
	extra := []*rpc.ExtraKeyVal{
		{
			Key: "agent.gitlab.com/id",
			Val: []string{strconv.FormatInt(auth.Agent.Id, 10)},
		},
		{
			Key: "agent.gitlab.com/username",
			Val: []string{auth.User.Username},
		},
		{
			Key: "agent.gitlab.com/access_type",
			Val: []string{accessType},
		},
		{
			Key: "agent.gitlab.com/config_project_id",
			Val: []string{strconv.FormatInt(auth.Agent.ConfigProject.Id, 10)},
		},
	}
	return extra
}
