# Copyright © The Debusine Developers
# See the AUTHORS file at the top-level directory of this distribution
#
# This file is part of Debusine. It is subject to the license terms
# in the LICENSE file found in the top-level directory of this
# distribution. No part of Debusine, including this file, may be copied,
# modified, propagated, or distributed except according to the terms
# contained in the LICENSE file.

"""Piuparts workflow."""

from typing import Any

from django.utils import timezone

from debusine.artifacts.models import (
    ArtifactCategory,
    DebianBinaryPackage,
    DebianUpload,
    get_source_package_version,
)
from debusine.client.models import LookupChildType
from debusine.db.models import Artifact, ArtifactRelation, WorkRequest
from debusine.server.collections.lookup import lookup_multiple
from debusine.server.workflows import workflow_utils
from debusine.server.workflows.models import (
    PiupartsWorkflowData,
    WorkRequestWorkflowData,
)
from debusine.server.workflows.regression_tracking import (
    RegressionTrackingWorkflow,
)
from debusine.tasks.models import (
    ActionUpdateCollectionWithArtifacts,
    BackendType,
    BaseDynamicTaskData,
    OutputData,
    PiupartsData,
    PiupartsDataInput,
    RegressionAnalysis,
)
from debusine.tasks.server import TaskDatabaseInterface


class PiupartsWorkflow(
    RegressionTrackingWorkflow[PiupartsWorkflowData, BaseDynamicTaskData]
):
    """
    Checks binary packages for all architectures of a target distribution.

    Creates piuparts tasks corresponding to the output of a set of
    build tasks for multiple architectures. This is meant to be part
    of a larger QA workflow such as Debian Pipeline.
    """

    # Workflow name (used by create-workflow-template)
    TASK_NAME = "piuparts"

    def post_init(self) -> None:
        """Initialize."""
        if self.data.backend == BackendType.AUTO:
            self.data.backend = BackendType.UNSHARE

    def _has_current_reference_qa_result(self, architecture: str) -> bool:
        """
        Return True iff we have a current reference QA result.

        A piuparts analysis is outdated if the underlying binary packages
        are outdated (i.e.  have smaller version numbers) compared to what's
        available in the ``debian:suite`` collection

        Otherwise, it is current.
        """
        # This method is only called when update_qa_results is True, in
        # which case these are checked by a model validator.
        assert self.qa_suite is not None
        assert self.reference_qa_results is not None

        source_data = workflow_utils.source_package_data(self)
        latest_result = self.reference_qa_results.manager.lookup(
            f"latest:piuparts_{source_data.name}_{architecture}"
        )
        return (
            latest_result is not None
            and latest_result.data["version"] == source_data.version
        )

    def _populate_single(self, architecture: str) -> None:
        """Create a Piuparts task for a single architecture."""
        if (
            self.data.update_qa_results
            and self._has_current_reference_qa_result(architecture)
        ):
            return

        # Define piuparts environment
        base_tgz = f"{self.data.vendor}/match:codename={self.data.codename}"
        environment = self.data.environment or base_tgz
        backend = self.data.backend

        # group arch-all + arch-any
        arch_subset_binary_artifacts = (
            workflow_utils.filter_artifact_lookup_by_arch(
                self, self.data.binary_artifacts, {architecture, "all"}
            )
        )

        # The concrete architecture, or {arch_all_build_architecture} if only
        # Architecture: all binary packages are being checked by this task
        build_architecture = (
            self.data.arch_all_build_architecture
            if architecture == "all"
            else architecture
        )

        extra_repositories = workflow_utils.configure_for_overlay_suite(
            self,
            extra_repositories=self.data.extra_repositories,
            vendor=self.data.vendor,
            codename=self.data.codename,
            environment=environment,
            backend=self.data.backend,
            architecture=build_architecture,
            try_variant="piuparts",
        )

        # Create work-request (idempotent)
        workflow_data_kwargs: dict[str, Any] = {}
        if self.data.update_qa_results:
            # When updating reference results for regression tracking, task
            # failures never cause the parent workflow or dependent tasks to
            # fail.
            workflow_data_kwargs["allow_failure"] = True
        wr = self.work_request_ensure_child_worker(
            task_name="piuparts",
            task_data=PiupartsData(
                backend=backend,
                environment=environment,
                input=PiupartsDataInput(
                    binary_artifacts=arch_subset_binary_artifacts
                ),
                build_architecture=build_architecture,
                base_tgz=base_tgz,
                extra_repositories=extra_repositories,
            ),
            workflow_data=WorkRequestWorkflowData(
                display_name=f"Piuparts {architecture}",
                step=f"piuparts-{architecture}",
                **workflow_data_kwargs,
            ),
        )

        # Any of the lookups in input.binary_artifacts may result in
        # promises, and in that case the workflow adds corresponding
        # dependencies. Binary promises must include an architecture field
        # in their data.
        self.requires_artifact(wr, arch_subset_binary_artifacts)
        promise_name = f"{self.data.prefix}piuparts-{architecture}"
        self.provides_artifact(wr, ArtifactCategory.PIUPARTS, promise_name)

        if self.data.update_qa_results:
            # Checked by a model validator.
            assert self.data.reference_qa_results is not None

            source_data = workflow_utils.source_package_data(self)

            # Back off if another workflow gets there first.
            self.skip_if_qa_result_changed(
                wr, package=source_data.name, architecture=architecture
            )

            # Record results in the reference collection.
            action = ActionUpdateCollectionWithArtifacts(
                collection=self.data.reference_qa_results,
                variables={
                    "package": source_data.name,
                    "version": source_data.version,
                    "architecture": architecture,
                    "timestamp": int(
                        (self.qa_suite_changed or timezone.now()).timestamp()
                    ),
                    "work_request_id": wr.id,
                },
                artifact_filters={"category": ArtifactCategory.PIUPARTS},
            )
            wr.add_event_reaction("on_success", action)
            wr.add_event_reaction("on_failure", action)

        if self.data.enable_regression_tracking:
            # Checked by a model validator.
            assert self.data.reference_prefix

            regression_analysis = self.work_request_ensure_child_internal(
                task_name="workflow",
                workflow_data=WorkRequestWorkflowData(
                    allow_dependency_failures=True,
                    step="regression-analysis",
                    display_name=f"Regression analysis for {architecture}",
                    visible=False,
                ),
            )
            try:
                self.requires_artifact(
                    regression_analysis,
                    f"internal@collections/name:"
                    f"{self.data.reference_prefix}piuparts-{architecture}",
                )
            except KeyError:
                pass
            regression_analysis.add_dependency(wr)
            self.orchestrate_child(regression_analysis)

    def populate(self) -> None:
        """Create a Piuparts task for each concrete architecture."""
        # piuparts will be run on the intersection of the provided
        # list of architectures (if any) and the architectures
        # provided in binary_artifacts
        architectures = workflow_utils.get_lookup_architectures(
            self, self.data.binary_artifacts
        )
        #  architectures: if set, only run on any of these architecture names
        if self.data.architectures is not None:
            architectures.intersection_update(self.data.architectures)

        if architectures == {"all"}:
            # If only Architecture: all binary packages are provided
            # in binary_artifacts, then piuparts will be run once for
            # arch-all on {arch_all_build_architecture}.
            pass
        else:
            # piuparts will be run grouping arch-all + arch-any
            # together. Not running "all" separately.
            architectures.discard("all")

        for architecture in sorted(architectures):
            self._populate_single(architecture)

    def _extract_artifact_details(
        self, artifact: Artifact | None
    ) -> tuple[str | None, int | None, WorkRequest | None]:
        """Extract details from a ``debian:piuparts`` artifact."""
        source_version: str | None
        artifact_id: int | None
        wr: WorkRequest | None
        if artifact is None:
            source_version = None
            artifact_id = None
            wr = None
        else:
            analyzed_artifact = (
                artifact.relations.filter(
                    type=ArtifactRelation.Relations.RELATES_TO
                )
                .earliest("id")
                .target
            )
            analyzed_data = analyzed_artifact.create_data()
            assert isinstance(
                analyzed_data,
                (DebianBinaryPackage, DebianUpload),
            )
            source_version = get_source_package_version(analyzed_data)
            artifact_id = artifact.id
            wr = artifact.created_by_work_request
        return source_version, artifact_id, wr

    def callback_regression_analysis(self) -> bool:
        """
        Analyze regressions compared to reference results.

        This is called once for each architecture, but updates the whole
        analysis for all architectures each time.  This is partly for
        simplicity and robustness (we don't need to work out how to combine
        the new analysis with a previous one), and partly to make it easier
        to handle cases where there isn't a one-to-one mapping between the
        reference results and the new results.
        """
        # Select the newest result for each architecture.
        reference_artifacts = self.find_reference_artifacts("piuparts")
        new_artifacts = self.find_new_artifacts(
            "piuparts", ArtifactCategory.PIUPARTS
        )

        output_data = self.work_request.output_data or OutputData()
        output_data.regression_analysis = {}
        for architecture in sorted(
            set(reference_artifacts) | set(new_artifacts)
        ):
            reference = reference_artifacts.get(architecture)
            reference_source_version, reference_artifact_id, reference_wr = (
                self._extract_artifact_details(reference)
            )
            new = new_artifacts.get(architecture)
            new_source_version, new_artifact_id, new_wr = (
                self._extract_artifact_details(new)
            )

            output_data.regression_analysis[architecture] = RegressionAnalysis(
                original_source_version=reference_source_version,
                original_artifact_id=reference_artifact_id,
                new_source_version=new_source_version,
                new_artifact_id=new_artifact_id,
                status=self.compare_qa_results(reference_wr, new_wr),
            )

        self.work_request.output_data = output_data
        self.work_request.save()
        return True

    def build_dynamic_data(
        self, task_database: TaskDatabaseInterface  # noqa: U100
    ) -> BaseDynamicTaskData:
        """
        Compute dynamic data for this workflow.

        :subject: source package names of ``binary_artifacts`` separated
          by spaces
        :parameter_summary: "subject" in ``vendor``:``codename``
        """
        binaries = lookup_multiple(
            self.data.binary_artifacts,
            workflow_root=self.work_request.workflow_root,
            workspace=self.workspace,
            user=self.work_request.created_by,
            expect_type=LookupChildType.ARTIFACT_OR_PROMISE,
        )
        source_package_names = " ".join(
            workflow_utils.get_source_package_names(
                binaries,
                configuration_key="binary_artifacts",
                artifact_expected_categories=(
                    ArtifactCategory.BINARY_PACKAGE,
                    ArtifactCategory.UPLOAD,
                ),
            )
        )

        return BaseDynamicTaskData(
            subject=source_package_names,
            parameter_summary=f"{source_package_names} in "
            f"{self.data.vendor}:{self.data.codename}",
        )
