// Copyright Earl Warren <contact@earl-warren.org>
// Copyright Loïc Dachary <loic@dachary.org>
// SPDX-License-Identifier: MIT

package generic

import (
	"context"

	"code.forgejo.org/f3/gof3/v3/f3"
	"code.forgejo.org/f3/gof3/v3/id"
	"code.forgejo.org/f3/gof3/v3/kind"
	"code.forgejo.org/f3/gof3/v3/logger"
	"code.forgejo.org/f3/gof3/v3/options"
	options_logger "code.forgejo.org/f3/gof3/v3/options/logger"
)

type Storage map[string]int

type testTree struct {
	Tree
}

type testOptions struct {
	options.Options
	options_logger.OptionsLogger
}

func newTestOptions() options.Interface {
	opts := &testOptions{}
	opts.SetName("test")
	l := logger.NewLogger()
	l.SetLevel(logger.Trace)
	opts.SetLogger(l)
	return opts
}

type noopNodeDriver struct {
	NullDriver
}

func (o *noopNodeDriver) ListPage(context.Context, int) ChildrenSlice {
	return ChildrenSlice{}
}

func (o *noopNodeDriver) GetIDFromName(ctx context.Context, name string) id.NodeID {
	return id.NilID
}

func (o *noopNodeDriver) Equals(context.Context, NodeInterface) bool {
	return false
}

func (o *noopNodeDriver) Put(context.Context) id.NodeID {
	return o.GetNode().GetID()
}

func (o *noopNodeDriver) Patch(context.Context) {
}

func (o *noopNodeDriver) NewFormat() f3.Interface {
	f := f3.NewCommon(id.NilID.String())
	return &f
}

func (o *noopNodeDriver) ToFormat() f3.Interface {
	f := f3.NewCommon(o.GetNode().GetID().String())
	return &f
}

func (o *noopNodeDriver) FromFormat(f f3.Interface) {
	o.GetNode().SetID(id.NewNodeID(f.GetID()))
}

func newTestNodeDriver() NodeDriverInterface {
	return &noopNodeDriver{}
}

type testTreeDriver struct {
	NullTreeDriver
}

func newTestTreeDriver() TreeDriverInterface {
	return &testTreeDriver{}
}

func (o *testTreeDriver) Factory(ctx context.Context, kind kind.Kind) NodeDriverInterface {
	d := newTestNodeDriver()
	d.SetTreeDriver(o)
	return d
}

func newTestTree() TreeInterface {
	tree := &testTree{}
	tree.Init(tree, newTestOptions())
	tree.Trace("init done")
	tree.SetDriver(newTestTreeDriver())
	tree.Register(kindTestNodeLevelOne, func(ctx context.Context, kind kind.Kind) NodeInterface {
		node := &testNodeLevelOne{}
		return node.Init(node)
	})
	tree.Register(kindTestNodeLevelTwo, func(ctx context.Context, kind kind.Kind) NodeInterface {
		node := &testNodeLevelTwo{}
		return node.Init(node)
	})
	return tree
}

type testNodeInterface interface {
	GetV() int
}

type testNode struct {
	Node
	v int
}

func (o *testNode) GetV() int {
	return o.v
}

func (o *testNode) Equals(ctx context.Context, other NodeInterface) bool {
	if !o.Node.Equals(ctx, other) {
		return false
	}
	return o.GetV() == other.(testNodeInterface).GetV()
}

var kindTestNodeLevelOne = kind.Kind("levelone")

type testNodeLevelOne struct {
	testNode
}

var kindTestNodeLevelTwo = kind.Kind("leveltwo")

type testNodeLevelTwo struct {
	testNode
}
