// Copyright (c) 2015, Daniel Martí <mvdan@mvdan.cc>
// See LICENSE for licensing information

package main

import (
	"log"
	"os"
	"sort"
	"strings"
	"text/template"

	"golang.org/x/net/idna"

	"github.com/mvdan/xurls"
)

const path = "regex.go"

var regexTmpl = template.Must(template.New("regex").Parse(`// Generated by regexgen

package xurls

const (
{{- range $key, $value := . }}
	{{$key}} = ` + "`" + `{{$value}}` + "`" + `
{{- end}}
)
`))

func writeRegex(tlds []string) error {
	allTldsSet := make(map[string]struct{})
	add := func(tld string) {
		if _, e := allTldsSet[tld]; e {
			log.Fatalf("Duplicate TLD: %s", tld)
		}
		allTldsSet[tld] = struct{}{}
	}
	for _, tldlist := range [...][]string{tlds, xurls.PseudoTLDs} {
		for _, tld := range tldlist {
			add(tld)
			asciiTld, err := idna.ToASCII(tld)
			if err != nil {
				return err
			}
			if asciiTld != tld {
				add(asciiTld)
			}
		}
	}
	var allTlds []string
	for tld := range allTldsSet {
		allTlds = append(allTlds, tld)
	}
	sort.Strings(allTlds)
	f, err := os.Create(path)
	if err != nil {
		return err
	}
	defer f.Close()
	return regexTmpl.Execute(f, map[string]string{
		"gtld       ": `(?i)(` + strings.Join(allTlds, `|`) + `)(?-i)`,
		"otherScheme": `(?i)(` + strings.Join(xurls.SchemesNoAuthority, `|`) + `)(?-i):`,
	})
}

func main() {
	log.Printf("Generating %s...", path)
	if err := writeRegex(xurls.TLDs); err != nil {
		log.Fatalf("Could not write %s: %v", path, err)
	}
}
