/* NVTV TV common routines -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id: tv_common.c,v 1.24 2004/01/27 17:05:25 dthierbach Exp $
 *
 * Contents:
 *
 * Header: Common tv-related routines.
 *
 */

#include "local.h" /* before everything else */

#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

#include "tv_i2c.h"
#include "tv_common.h"
#include "tv_bt.h"
#include "tv_cx.h"
#include "tv_ch1_7007.h"
#include "tv_ch2_7009.h"
#include "tv_ph1_saa7102.h"
#include "tv_ph2_saa7104.h"
#include "tv_nx.h"
#include "tv_null.h"

/* -------- */

TVState tvState = TV_UNKNOWN;

/* -------- */

/*
 * Detect devices that share the 0x88/0x8a address, i.e. BT/CX/PH/TW
 *
 */

TVChip 
TVDetectDeviceA (I2CDevPtr dev)
{
  I2CByte x1, x2, y;

  RAISE (MSG_DEBUG, "TVDetectDeviceA (%1s:%02X)", I2C_ID(dev));
#ifdef FAKE_PROBE_ID
  return FAKE_PROBE_ID;
#endif
  tvBusOk = TRUE;  
  TVWriteBus (dev, 0xff, 0x00);
  RAISE (MSG_DEBUG, "  check LUT (%s)", tvBusOk ? "ok" : "err");
  if (!tvBusOk) return TV_BROOKTREE;

  /* So it's not a BT, but may be an CX, in either read mode, or a PH or TW. 
   * Check register 01. It's read only for CX/TW, and reserved for PH, 
   * with default value 00. Although it seems to be used for Macrovision,
   * it's either not writeable or not readable.
   */

  TVWriteBus (dev, 0x01, 0x00); 
  TVReadBus  (dev, 0x00, &x1); 
  TVReadBus  (dev, 0x01, &y); 
  TVReadBus  (dev, 0x00, &x2); 
  RAISE (MSG_DEBUG, "  a %02X/%02X %02X (%s)", x1, x2, y, 
	 tvBusOk ? "ok" : "err");
  if (!tvBusOk) return TV_BROOKTREE;
  
  if (y == 0x00) {

    /* It's a PH, TW, or CX in status mode.
     * For PH or TW, reg 0x00 is never zero, so it must be a CX. 
     * Otherwise, test register 1a, which is msm threshold for PH, and
     * a status register for the TW, with at least the lower 4 bits r/o.
     */

    if (x1 == 0x00 || x2 == 0x00) return TV_CONEXANT;
    TVReadBus (dev, 0x1a, &x1); 
    x2 = x1 ^ 0x0f;
    TVWriteBus (dev, 0x1a, x2);
    TVReadBus  (dev, 0x1a, &y); 
    TVWriteBus (dev, 0x1a, x1);
    RAISE (MSG_DEBUG, "  b %02X/%02X %02X (%s)", x1, x2, y, 
	   tvBusOk ? "ok" : "err");
    if (!tvBusOk) return TV_NO_CHIP;
    if (y == x2) return TV_PHILIPS;
    return TV_NO_CHIP;

  } else {

    /* It's a TW or CX in read mode.  Check register 04 bit 1. For
     * the CX, this is the PAL status flag (r/o), which shouldn't
     * change frequently. However, the other bits may change. For
     * both TW98 and TW99, it is the horizontal sync length HSLEN.
     * Changing it slightly and then resetting it shouldn't hurt.
     */

    TVReadBus (dev, 0x04, &y);
    TVWriteBus (dev, 0x04, y ^ 0x02); /* Bit 2, CX: PAL, TW: HSLEN */
    TVReadBus (dev, 0x04, &x1);
    TVWriteBus (dev, 0x04, y); 
    TVReadBus (dev, 0x04, &x2);
    RAISE (MSG_DEBUG, "  c %02X/%02X/%02X (%s)", y, x1, x2, 
	   tvBusOk ? "ok" : "err");

    if (!tvBusOk) return TV_NO_CHIP;
    if (x1 == (y ^ 0x02) && x2 == y) return TV_NO_CHIP;
    return TV_CONEXANT;
  }
  return TV_NO_CHIP;
}

/*
 * Detect devices that share the 0xEA/0xEC address, i.e. CH types
 *
 */

TVChip 
TVDetectDeviceB (I2CDevPtr dev)
{
  I2CByte x1, x2;

  RAISE (MSG_DEBUG, "TVDetectDeviceB (%1s:%02X)", I2C_ID(dev));
#ifdef FAKE_PROBE_ID
  return FAKE_PROBE_ID;
#endif
  tvBusOk = TRUE;  
  TVReadBus  (dev, 0xc0|0x0a, &x1);
  TVWriteBus (dev, 0xc0|0x0a, x1 ^ 1);
  TVReadBus  (dev, 0xc0|0x0a, &x2);
  TVWriteBus (dev, 0xc0|0x0a, x1);
  if (x1 == x2) {
    return TV_CHRONTEL_MODEL2;
  } else {
    return TV_CHRONTEL_MODEL1;
  }
}

/* 

Write to 0x4a/0x4b. (0x0a is hpr for 7007).

7009: Won't change
7007: Will change, so write original values back.

*/

/* -------- */

I2CChainPtr 
TVFindDevice (I2CChainPtr root, TVChip chip_type)
{
  I2CChainPtr chain;

  for (chain = root; chain; chain = chain->next)
  {
    if (chain->type == chip_type) return chain;
  }
  return NULL;
}

/* FIXME: NVFindTvDevice -> TVFindDevice (pNv->TvChain, ...) */

I2CChainPtr 
TVAllocChainEntry (I2CChainPtr root, I2CDevPtr dev, TVChip chip_type, 
  char* name)
{
  I2CChainPtr chain;

  chain = xcalloc (1, sizeof (I2CChainRec));
  chain->dev = dev;
  chain->next = root;
  chain->type = chip_type;
  chain->name = xalloc (strlen(name)+1);
  strcpy (chain->name, name);
  return chain;
}

/* 
 * Create chain of tv chips from list of devices on busses.
 * Process higher numbered busses first. FIXME??
 * If 'all' is true, include all
 * devices, otherwise only correctly detected ones.
 */

I2CChainPtr 
TVCreateChain (I2CBusPtr busses[], int nbus, I2CChainPtr chain, Bool all)
{
  int bus;
  I2CDevPtr dev;
  I2CChainPtr root;
  char version[50];
  char *s;
  TVChip chip;

  RAISE (MSG_DEBUG, "TVCreateChain %i %i", nbus, all);
  root = chain;
  for (bus = 0; bus < nbus; bus++) {
    for (dev = busses[bus]->FirstDev; dev; dev = dev->NextDev) {
      s = NULL;
      chip = TV_NO_CHIP;
      switch (dev->SlaveAddr) {
	case 0x88: 
        case 0x8a: /* FIXME: For test only */
	  chip = TVDetectDeviceA (dev);
	  break;
	case 0xea:
	case 0xec:
	  chip = TVDetectDeviceB (dev);
	  break;
      }
      switch (chip & TV_ENCODER) {
	case TV_CHRONTEL_MODEL1:
	  s = TVDetectChrontel1 (dev, &chip);
	  break;
	case TV_CHRONTEL_MODEL2:
	  s = TVDetectChrontel2 (dev, &chip);
	  break;
        case TV_PHILIPS:
	  s = TVDetectPhilips (dev, &chip);
	  break; 
	case TV_CONEXANT:
	  s = TVDetectConexant (dev, &chip);
	  break;
	case TV_BROOKTREE:
	  s = TVDetectBrooktree (dev, &chip, tvState);
	  break; 
        default:
	  if (!all) break;
	  snprintf (version, 50, "Unknown chip (%1s:%02X)", I2C_ID(dev));
	  s = version;
      }
      if (s) root = TVAllocChainEntry (root, dev, chip, s);
    }
  }
  return root;
}

void 
TVProbeDevice (I2CBusPtr bus, I2CSlaveAddr addr, char *format, ...)
{
  I2CDevPtr dev;
  char *s;
  va_list ap;

#ifndef FAKE_PROBE_ALL
#ifndef FAKE_PROBE_ADDR
  if (TVProbeBus (bus, addr))
#else
  if (addr == FAKE_PROBE_ADDR)
#endif
#endif
  {
    dev = xf86CreateI2CDevRec();
    s = xalloc (8); 
    va_start (ap, format);
    vsnprintf (s, 7, format, ap);
    va_end (ap);
    dev->DevName = s;
    dev->SlaveAddr = addr;
    dev->pI2CBus = bus;
    if (!xf86I2CDevInit(dev)) {
      xfree (dev->DevName); 
      xf86DestroyI2CDevRec(dev, TRUE);
    }
  }
}

void 
TVProbeKnownDevices (I2CBusPtr busses[], int nbus)
{
  int bus;
  
  for (bus = 0; bus < nbus; bus++) {
    TVProbeDevice (busses[bus], 0x88, "%i:%02X", bus, 0x88);
    TVProbeDevice (busses[bus], 0x8A, "%i:%02X", bus, 0x8A);
    TVProbeDevice (busses[bus], 0xEA, "%i:%02X", bus, 0xEA);
    TVProbeDevice (busses[bus], 0xEC, "%i:%02X", bus, 0xEC);
  }
}

void 
TVProbeAllDevices (I2CBusPtr busses[], int nbus)
{
  I2CSlaveAddr addr;
  int bus;

  for (bus = 0; bus < nbus; bus++) {
    for (addr = 0x00; addr < 0x100; addr += 2) {
      TVProbeDevice (busses[bus], addr, "%1i:%02X", bus, addr);
    }
  }
}

void 
TVCheckChain (I2CChainPtr chain)
{
  if (chain) RAISE (MSG_WARNING, "TVCheckChain: Memory leak.");
}

I2CChainPtr
TVProbeCreateAll (I2CBusPtr busses[], int nbus, I2CChainPtr chain)
{
  TVProbeAllDevices (busses, nbus);
  return TVCreateChain (busses, nbus, chain, TRUE);
}

I2CChainPtr
TVProbeCreateKnown (I2CBusPtr busses[], int nbus, I2CChainPtr chain)
{
  TVProbeKnownDevices (busses, nbus); 
  chain = TVCreateChain (busses, nbus, chain, FALSE);
#ifdef PROBE_ALL_UNKNOWN
  /* if no dev's found, scan all addresses */
  if (!chain) {
    TVProbeAllDevices (busses, nbus);
    chain = TVCreateChain (busses, nbus, NULL, TRUE);
  }
#endif
  return chain;
}

/* ---- destroy routines ---- */

// NVDestroyDevices == devices & chain 
// NVDestroyBusses (NVPtr pNv) == devices, chain, all busses

void 
TVDestroyDevices (I2CBusPtr busses[], int nbus)
{
  I2CDevPtr dev;
  int bus;

  for (bus = 0; bus < nbus; bus++) {
    if (busses[bus]) {
      while ((dev = busses[bus]->FirstDev) != NULL) {
	xfree (dev->DevName);
	xf86DestroyI2CDevRec(dev, TRUE); 
      }
    }
  }
}

/*
 * Free the chain of tv chips
 */

void 
TVDestroyChain (I2CChainPtr root)
{
  I2CChainPtr chain;

  while (root) {
    xfree (root->name);
    chain = root->next;
    xfree (root);
    root = chain;
  }
}


void
TVDestroyChainDevices (I2CChainPtr *chain, I2CBusPtr busses[], int nbus)
{
  if (*chain) TVDestroyChain (*chain);
  *chain = NULL;
  TVDestroyDevices (busses, nbus);
}

void 
TVDestroyBus (I2CBusPtr bus)
{
  if (bus) {
    xfree (bus->DriverPrivate.ptr);
    xf86DestroyI2CBusRec (bus, TRUE, FALSE);
  }
}

void 
TVDestroyBusses (I2CBusPtr busses[], int nbus)
{
  int i;

  for (i = 0; i < nbus; i++) 
  {
    TVDestroyBus (busses[i]);
    busses[i] = NULL;
  }
}

void 
TVDestroyAll (I2CChainPtr *chain, I2CBusPtr busses[], int nbus)
{
  TVDestroyChainDevices (chain, busses, nbus);
  TVDestroyBusses (busses, nbus);
}

/* ---- ---- */

void 
TVSetTvEncoder (TVEncoderObj *encoder, I2CChainPtr chain, void* extra)
{
  void* ctrl = (void *) chain->dev; 

  switch (chain->type & TV_ENCODER) {
    case TV_BROOKTREE:
      *encoder = tvBtTemplate;
      break;
    case TV_CONEXANT:
      *encoder = tvCxTemplate;
      break;
    case TV_CHRONTEL_MODEL1:
      *encoder = tvCh1Template;
      break;
    case TV_CHRONTEL_MODEL2:
      *encoder = tvCh2Template;
      break;
    case TV_PHILIPS_MODEL1:
      *encoder = tvPh1Template;
      break;
    case TV_PHILIPS_MODEL2:
      *encoder = tvPh2Template;
      break;
    case TV_NVIDIA:
      *encoder = tvNxTemplate;
      ctrl = extra;
      break;
    default:
      *encoder = tvNullTemplate;
      break;
  }
  encoder->Create (encoder, chain->type, ctrl);
}

