# pylint: disable=too-many-lines
# coding=utf-8
# --------------------------------------------------------------------------
# Copyright (c) Microsoft Corporation. All rights reserved.
# Licensed under the MIT License. See License.txt in the project root for license information.
# Code generated by Microsoft (R) AutoRest Code Generator.
# Changes may cause incorrect behavior and will be lost if the code is regenerated.
# --------------------------------------------------------------------------
from io import IOBase
import sys
from typing import Any, AsyncIterable, Callable, Dict, IO, List, Optional, TypeVar, Union, cast, overload
import urllib.parse

from azure.core.async_paging import AsyncItemPaged, AsyncList
from azure.core.exceptions import (
    ClientAuthenticationError,
    HttpResponseError,
    ResourceExistsError,
    ResourceNotFoundError,
    ResourceNotModifiedError,
    map_error,
)
from azure.core.pipeline import PipelineResponse
from azure.core.rest import AsyncHttpResponse, HttpRequest
from azure.core.tracing.decorator import distributed_trace
from azure.core.tracing.decorator_async import distributed_trace_async
from azure.core.utils import case_insensitive_dict

from ...operations._operations import (
    build_approval_approve_request,
    build_approval_reject_request,
    build_task_status_update_request,
    build_user_requests_submit_request,
    build_workflow_create_or_replace_request,
    build_workflow_delete_request,
    build_workflow_get_request,
    build_workflow_run_cancel_request,
    build_workflow_run_get_request,
    build_workflow_runs_list_request,
    build_workflow_task_get_request,
    build_workflow_task_reassign_request,
    build_workflow_tasks_list_request,
    build_workflow_validate_request,
    build_workflows_list_request,
)

if sys.version_info >= (3, 9):
    from collections.abc import MutableMapping
else:
    from typing import MutableMapping  # type: ignore
JSON = MutableMapping[str, Any]  # pylint: disable=unsubscriptable-object
T = TypeVar("T")
ClsType = Optional[Callable[[PipelineResponse[HttpRequest, AsyncHttpResponse], T, Dict[str, Any]], Any]]


class WorkflowsOperations:
    """
    .. warning::
        **DO NOT** instantiate this class directly.

        Instead, you should access the following operations through
        :class:`~azure.purview.workflow.aio.PurviewWorkflowClient`'s
        :attr:`workflows` attribute.
    """

    def __init__(self, *args, **kwargs) -> None:
        input_args = list(args)
        self._client = input_args.pop(0) if input_args else kwargs.pop("client")
        self._config = input_args.pop(0) if input_args else kwargs.pop("config")
        self._serialize = input_args.pop(0) if input_args else kwargs.pop("serializer")
        self._deserialize = input_args.pop(0) if input_args else kwargs.pop("deserializer")

    @distributed_trace
    def list(self, **kwargs: Any) -> AsyncIterable[JSON]:
        """List all workflows.

        :return: An iterator like instance of JSON object
        :rtype: ~azure.core.async_paging.AsyncItemPaged[JSON]
        :raises ~azure.core.exceptions.HttpResponseError:

        Example:
            .. code-block:: python

                # response body for status code(s): 200
                response == {
                    "description": "str",
                    "id": "str",
                    "isEnabled": bool,
                    "name": "str",
                    "triggers": [
                        {
                            "type": "str",
                            "underCollection": "str",
                            "underGlossary": "str",
                            "underGlossaryHierarchy": "str"
                        }
                    ],
                    "createdBy": "str",
                    "createdTime": "2020-02-20 00:00:00",
                    "lastUpdateTime": "2020-02-20 00:00:00",
                    "updatedBy": "str"
                }
        """
        _headers = kwargs.pop("headers", {}) or {}
        _params = kwargs.pop("params", {}) or {}

        cls: ClsType[JSON] = kwargs.pop("cls", None)

        error_map: MutableMapping = {
            401: ClientAuthenticationError,
            404: ResourceNotFoundError,
            409: ResourceExistsError,
            304: ResourceNotModifiedError,
        }
        error_map.update(kwargs.pop("error_map", {}) or {})

        def prepare_request(next_link=None):
            if not next_link:

                _request = build_workflows_list_request(
                    api_version=self._config.api_version,
                    headers=_headers,
                    params=_params,
                )
                path_format_arguments = {
                    "endpoint": self._serialize.url(
                        "self._config.endpoint", self._config.endpoint, "str", skip_quote=True
                    ),
                }
                _request.url = self._client.format_url(_request.url, **path_format_arguments)

            else:
                # make call to next link with the client's api-version
                _parsed_next_link = urllib.parse.urlparse(next_link)
                _next_request_params = case_insensitive_dict(
                    {
                        key: [urllib.parse.quote(v) for v in value]
                        for key, value in urllib.parse.parse_qs(_parsed_next_link.query).items()
                    }
                )
                _next_request_params["api-version"] = self._config.api_version
                _request = HttpRequest(
                    "GET", urllib.parse.urljoin(next_link, _parsed_next_link.path), params=_next_request_params
                )
                path_format_arguments = {
                    "endpoint": self._serialize.url(
                        "self._config.endpoint", self._config.endpoint, "str", skip_quote=True
                    ),
                }
                _request.url = self._client.format_url(_request.url, **path_format_arguments)

            return _request

        async def extract_data(pipeline_response):
            deserialized = pipeline_response.http_response.json()
            list_of_elem = deserialized["value"]
            if cls:
                list_of_elem = cls(list_of_elem)  # type: ignore
            return deserialized.get("nextLink") or None, AsyncList(list_of_elem)

        async def get_next(next_link=None):
            _request = prepare_request(next_link)

            _stream = False
            pipeline_response: PipelineResponse = await self._client._pipeline.run(  # pylint: disable=protected-access
                _request, stream=_stream, **kwargs
            )
            response = pipeline_response.http_response

            if response.status_code not in [200]:
                map_error(status_code=response.status_code, response=response, error_map=error_map)
                raise HttpResponseError(response=response)

            return pipeline_response

        return AsyncItemPaged(get_next, extract_data)


class WorkflowOperations:
    """
    .. warning::
        **DO NOT** instantiate this class directly.

        Instead, you should access the following operations through
        :class:`~azure.purview.workflow.aio.PurviewWorkflowClient`'s
        :attr:`workflow` attribute.
    """

    def __init__(self, *args, **kwargs) -> None:
        input_args = list(args)
        self._client = input_args.pop(0) if input_args else kwargs.pop("client")
        self._config = input_args.pop(0) if input_args else kwargs.pop("config")
        self._serialize = input_args.pop(0) if input_args else kwargs.pop("serializer")
        self._deserialize = input_args.pop(0) if input_args else kwargs.pop("deserializer")

    @distributed_trace_async
    async def get(self, workflow_id: str, **kwargs: Any) -> JSON:
        """Get a specific workflow.

        :param workflow_id: The workflow id. Required.
        :type workflow_id: str
        :return: JSON object
        :rtype: JSON
        :raises ~azure.core.exceptions.HttpResponseError:

        Example:
            .. code-block:: python

                # response body for status code(s): 200
                response == {
                    "actionDag": {},
                    "description": "str",
                    "id": "str",
                    "isEnabled": bool,
                    "name": "str",
                    "triggers": [
                        {
                            "type": "str",
                            "underCollection": "str",
                            "underGlossary": "str",
                            "underGlossaryHierarchy": "str"
                        }
                    ],
                    "createdBy": "str",
                    "createdTime": "2020-02-20 00:00:00",
                    "lastUpdateTime": "2020-02-20 00:00:00",
                    "updatedBy": "str"
                }
        """
        error_map: MutableMapping = {
            401: ClientAuthenticationError,
            404: ResourceNotFoundError,
            409: ResourceExistsError,
            304: ResourceNotModifiedError,
        }
        error_map.update(kwargs.pop("error_map", {}) or {})

        _headers = kwargs.pop("headers", {}) or {}
        _params = kwargs.pop("params", {}) or {}

        cls: ClsType[JSON] = kwargs.pop("cls", None)

        _request = build_workflow_get_request(
            workflow_id=workflow_id,
            api_version=self._config.api_version,
            headers=_headers,
            params=_params,
        )
        path_format_arguments = {
            "endpoint": self._serialize.url("self._config.endpoint", self._config.endpoint, "str", skip_quote=True),
        }
        _request.url = self._client.format_url(_request.url, **path_format_arguments)

        _stream = False
        pipeline_response: PipelineResponse = await self._client._pipeline.run(  # pylint: disable=protected-access
            _request, stream=_stream, **kwargs
        )

        response = pipeline_response.http_response

        if response.status_code not in [200]:
            map_error(status_code=response.status_code, response=response, error_map=error_map)
            raise HttpResponseError(response=response)

        if response.content:
            deserialized = response.json()
        else:
            deserialized = None

        if cls:
            return cls(pipeline_response, cast(JSON, deserialized), {})  # type: ignore

        return cast(JSON, deserialized)  # type: ignore

    @overload
    async def create_or_replace(
        self,
        workflow_id: str,
        workflow_create_or_update_command: JSON,
        *,
        content_type: str = "application/json",
        **kwargs: Any
    ) -> JSON:
        """Create or replace a workflow.

        :param workflow_id: The workflow id. Required.
        :type workflow_id: str
        :param workflow_create_or_update_command: Create or update workflow payload. Required.
        :type workflow_create_or_update_command: JSON
        :keyword content_type: Body Parameter content-type. Content type parameter for JSON body.
         Default value is "application/json".
        :paramtype content_type: str
        :return: JSON object
        :rtype: JSON
        :raises ~azure.core.exceptions.HttpResponseError:

        Example:
            .. code-block:: python

                # JSON input template you can fill out and use as your body input.
                workflow_create_or_update_command = {
                    "description": "str",
                    "isEnabled": bool,
                    "name": "str",
                    "triggers": [
                        {
                            "type": "str",
                            "underCollection": "str",
                            "underGlossary": "str",
                            "underGlossaryHierarchy": "str"
                        }
                    ],
                    "actionDag": {}
                }

                # response body for status code(s): 200
                response == {
                    "actionDag": {},
                    "description": "str",
                    "id": "str",
                    "isEnabled": bool,
                    "name": "str",
                    "triggers": [
                        {
                            "type": "str",
                            "underCollection": "str",
                            "underGlossary": "str",
                            "underGlossaryHierarchy": "str"
                        }
                    ],
                    "createdBy": "str",
                    "createdTime": "2020-02-20 00:00:00",
                    "lastUpdateTime": "2020-02-20 00:00:00",
                    "updatedBy": "str"
                }
        """

    @overload
    async def create_or_replace(
        self,
        workflow_id: str,
        workflow_create_or_update_command: IO[bytes],
        *,
        content_type: str = "application/json",
        **kwargs: Any
    ) -> JSON:
        """Create or replace a workflow.

        :param workflow_id: The workflow id. Required.
        :type workflow_id: str
        :param workflow_create_or_update_command: Create or update workflow payload. Required.
        :type workflow_create_or_update_command: IO[bytes]
        :keyword content_type: Body Parameter content-type. Content type parameter for binary body.
         Default value is "application/json".
        :paramtype content_type: str
        :return: JSON object
        :rtype: JSON
        :raises ~azure.core.exceptions.HttpResponseError:

        Example:
            .. code-block:: python

                # response body for status code(s): 200
                response == {
                    "actionDag": {},
                    "description": "str",
                    "id": "str",
                    "isEnabled": bool,
                    "name": "str",
                    "triggers": [
                        {
                            "type": "str",
                            "underCollection": "str",
                            "underGlossary": "str",
                            "underGlossaryHierarchy": "str"
                        }
                    ],
                    "createdBy": "str",
                    "createdTime": "2020-02-20 00:00:00",
                    "lastUpdateTime": "2020-02-20 00:00:00",
                    "updatedBy": "str"
                }
        """

    @distributed_trace_async
    async def create_or_replace(
        self, workflow_id: str, workflow_create_or_update_command: Union[JSON, IO[bytes]], **kwargs: Any
    ) -> JSON:
        """Create or replace a workflow.

        :param workflow_id: The workflow id. Required.
        :type workflow_id: str
        :param workflow_create_or_update_command: Create or update workflow payload. Is either a JSON
         type or a IO[bytes] type. Required.
        :type workflow_create_or_update_command: JSON or IO[bytes]
        :return: JSON object
        :rtype: JSON
        :raises ~azure.core.exceptions.HttpResponseError:

        Example:
            .. code-block:: python

                # JSON input template you can fill out and use as your body input.
                workflow_create_or_update_command = {
                    "description": "str",
                    "isEnabled": bool,
                    "name": "str",
                    "triggers": [
                        {
                            "type": "str",
                            "underCollection": "str",
                            "underGlossary": "str",
                            "underGlossaryHierarchy": "str"
                        }
                    ],
                    "actionDag": {}
                }

                # response body for status code(s): 200
                response == {
                    "actionDag": {},
                    "description": "str",
                    "id": "str",
                    "isEnabled": bool,
                    "name": "str",
                    "triggers": [
                        {
                            "type": "str",
                            "underCollection": "str",
                            "underGlossary": "str",
                            "underGlossaryHierarchy": "str"
                        }
                    ],
                    "createdBy": "str",
                    "createdTime": "2020-02-20 00:00:00",
                    "lastUpdateTime": "2020-02-20 00:00:00",
                    "updatedBy": "str"
                }
        """
        error_map: MutableMapping = {
            401: ClientAuthenticationError,
            404: ResourceNotFoundError,
            409: ResourceExistsError,
            304: ResourceNotModifiedError,
        }
        error_map.update(kwargs.pop("error_map", {}) or {})

        _headers = case_insensitive_dict(kwargs.pop("headers", {}) or {})
        _params = kwargs.pop("params", {}) or {}

        content_type: Optional[str] = kwargs.pop("content_type", _headers.pop("Content-Type", None))
        cls: ClsType[JSON] = kwargs.pop("cls", None)

        content_type = content_type or "application/json"
        _json = None
        _content = None
        if isinstance(workflow_create_or_update_command, (IOBase, bytes)):
            _content = workflow_create_or_update_command
        else:
            _json = workflow_create_or_update_command

        _request = build_workflow_create_or_replace_request(
            workflow_id=workflow_id,
            content_type=content_type,
            api_version=self._config.api_version,
            json=_json,
            content=_content,
            headers=_headers,
            params=_params,
        )
        path_format_arguments = {
            "endpoint": self._serialize.url("self._config.endpoint", self._config.endpoint, "str", skip_quote=True),
        }
        _request.url = self._client.format_url(_request.url, **path_format_arguments)

        _stream = False
        pipeline_response: PipelineResponse = await self._client._pipeline.run(  # pylint: disable=protected-access
            _request, stream=_stream, **kwargs
        )

        response = pipeline_response.http_response

        if response.status_code not in [200]:
            map_error(status_code=response.status_code, response=response, error_map=error_map)
            raise HttpResponseError(response=response)

        if response.content:
            deserialized = response.json()
        else:
            deserialized = None

        if cls:
            return cls(pipeline_response, cast(JSON, deserialized), {})  # type: ignore

        return cast(JSON, deserialized)  # type: ignore

    @distributed_trace_async
    async def delete(self, workflow_id: str, **kwargs: Any) -> None:
        """Delete a workflow.

        :param workflow_id: The workflow id. Required.
        :type workflow_id: str
        :return: None
        :rtype: None
        :raises ~azure.core.exceptions.HttpResponseError:
        """
        error_map: MutableMapping = {
            401: ClientAuthenticationError,
            404: ResourceNotFoundError,
            409: ResourceExistsError,
            304: ResourceNotModifiedError,
        }
        error_map.update(kwargs.pop("error_map", {}) or {})

        _headers = kwargs.pop("headers", {}) or {}
        _params = kwargs.pop("params", {}) or {}

        cls: ClsType[None] = kwargs.pop("cls", None)

        _request = build_workflow_delete_request(
            workflow_id=workflow_id,
            api_version=self._config.api_version,
            headers=_headers,
            params=_params,
        )
        path_format_arguments = {
            "endpoint": self._serialize.url("self._config.endpoint", self._config.endpoint, "str", skip_quote=True),
        }
        _request.url = self._client.format_url(_request.url, **path_format_arguments)

        _stream = False
        pipeline_response: PipelineResponse = await self._client._pipeline.run(  # pylint: disable=protected-access
            _request, stream=_stream, **kwargs
        )

        response = pipeline_response.http_response

        if response.status_code not in [204]:
            map_error(status_code=response.status_code, response=response, error_map=error_map)
            raise HttpResponseError(response=response)

        if cls:
            return cls(pipeline_response, None, {})  # type: ignore

    @overload
    async def validate(
        self, workflow_id: str, workflow_validate_query: JSON, *, content_type: str = "application/json", **kwargs: Any
    ) -> JSON:
        """Validate a workflow.

        :param workflow_id: The workflow id. Required.
        :type workflow_id: str
        :param workflow_validate_query: Check workflow payload. Required.
        :type workflow_validate_query: JSON
        :keyword content_type: Body Parameter content-type. Content type parameter for JSON body.
         Default value is "application/json".
        :paramtype content_type: str
        :return: JSON object
        :rtype: JSON
        :raises ~azure.core.exceptions.HttpResponseError:

        Example:
            .. code-block:: python

                # JSON input template you can fill out and use as your body input.
                workflow_validate_query = {
                    "description": "str",
                    "isEnabled": bool,
                    "name": "str",
                    "triggers": [
                        {
                            "type": "str",
                            "underCollection": "str",
                            "underGlossary": "str",
                            "underGlossaryHierarchy": "str"
                        }
                    ],
                    "actionDag": {}
                }

                # response body for status code(s): 200
                response == {
                    "value": [
                        {
                            "location": {
                                "type": "str",
                                "actionName": "str",
                                "parameterKey": "str"
                            },
                            "message": "str",
                            "severity": "str"
                        }
                    ]
                }
        """

    @overload
    async def validate(
        self,
        workflow_id: str,
        workflow_validate_query: IO[bytes],
        *,
        content_type: str = "application/json",
        **kwargs: Any
    ) -> JSON:
        """Validate a workflow.

        :param workflow_id: The workflow id. Required.
        :type workflow_id: str
        :param workflow_validate_query: Check workflow payload. Required.
        :type workflow_validate_query: IO[bytes]
        :keyword content_type: Body Parameter content-type. Content type parameter for binary body.
         Default value is "application/json".
        :paramtype content_type: str
        :return: JSON object
        :rtype: JSON
        :raises ~azure.core.exceptions.HttpResponseError:

        Example:
            .. code-block:: python

                # response body for status code(s): 200
                response == {
                    "value": [
                        {
                            "location": {
                                "type": "str",
                                "actionName": "str",
                                "parameterKey": "str"
                            },
                            "message": "str",
                            "severity": "str"
                        }
                    ]
                }
        """

    @distributed_trace_async
    async def validate(self, workflow_id: str, workflow_validate_query: Union[JSON, IO[bytes]], **kwargs: Any) -> JSON:
        """Validate a workflow.

        :param workflow_id: The workflow id. Required.
        :type workflow_id: str
        :param workflow_validate_query: Check workflow payload. Is either a JSON type or a IO[bytes]
         type. Required.
        :type workflow_validate_query: JSON or IO[bytes]
        :return: JSON object
        :rtype: JSON
        :raises ~azure.core.exceptions.HttpResponseError:

        Example:
            .. code-block:: python

                # JSON input template you can fill out and use as your body input.
                workflow_validate_query = {
                    "description": "str",
                    "isEnabled": bool,
                    "name": "str",
                    "triggers": [
                        {
                            "type": "str",
                            "underCollection": "str",
                            "underGlossary": "str",
                            "underGlossaryHierarchy": "str"
                        }
                    ],
                    "actionDag": {}
                }

                # response body for status code(s): 200
                response == {
                    "value": [
                        {
                            "location": {
                                "type": "str",
                                "actionName": "str",
                                "parameterKey": "str"
                            },
                            "message": "str",
                            "severity": "str"
                        }
                    ]
                }
        """
        error_map: MutableMapping = {
            401: ClientAuthenticationError,
            404: ResourceNotFoundError,
            409: ResourceExistsError,
            304: ResourceNotModifiedError,
        }
        error_map.update(kwargs.pop("error_map", {}) or {})

        _headers = case_insensitive_dict(kwargs.pop("headers", {}) or {})
        _params = kwargs.pop("params", {}) or {}

        content_type: Optional[str] = kwargs.pop("content_type", _headers.pop("Content-Type", None))
        cls: ClsType[JSON] = kwargs.pop("cls", None)

        content_type = content_type or "application/json"
        _json = None
        _content = None
        if isinstance(workflow_validate_query, (IOBase, bytes)):
            _content = workflow_validate_query
        else:
            _json = workflow_validate_query

        _request = build_workflow_validate_request(
            workflow_id=workflow_id,
            content_type=content_type,
            api_version=self._config.api_version,
            json=_json,
            content=_content,
            headers=_headers,
            params=_params,
        )
        path_format_arguments = {
            "endpoint": self._serialize.url("self._config.endpoint", self._config.endpoint, "str", skip_quote=True),
        }
        _request.url = self._client.format_url(_request.url, **path_format_arguments)

        _stream = False
        pipeline_response: PipelineResponse = await self._client._pipeline.run(  # pylint: disable=protected-access
            _request, stream=_stream, **kwargs
        )

        response = pipeline_response.http_response

        if response.status_code not in [200]:
            map_error(status_code=response.status_code, response=response, error_map=error_map)
            raise HttpResponseError(response=response)

        if response.content:
            deserialized = response.json()
        else:
            deserialized = None

        if cls:
            return cls(pipeline_response, cast(JSON, deserialized), {})  # type: ignore

        return cast(JSON, deserialized)  # type: ignore


class UserRequestsOperations:
    """
    .. warning::
        **DO NOT** instantiate this class directly.

        Instead, you should access the following operations through
        :class:`~azure.purview.workflow.aio.PurviewWorkflowClient`'s
        :attr:`user_requests` attribute.
    """

    def __init__(self, *args, **kwargs) -> None:
        input_args = list(args)
        self._client = input_args.pop(0) if input_args else kwargs.pop("client")
        self._config = input_args.pop(0) if input_args else kwargs.pop("config")
        self._serialize = input_args.pop(0) if input_args else kwargs.pop("serializer")
        self._deserialize = input_args.pop(0) if input_args else kwargs.pop("deserializer")

    @overload
    async def submit(
        self, user_requests_payload: JSON, *, content_type: str = "application/json", **kwargs: Any
    ) -> JSON:
        """Submit a user request for requestor, a user  request describes user ask to do operation(s) on
        Purview. If any workflow's trigger matches with an operation in request, a run of the workflow
        is created.

        :param user_requests_payload: The payload of submitting a user request. Required.
        :type user_requests_payload: JSON
        :keyword content_type: Body Parameter content-type. Content type parameter for JSON body.
         Default value is "application/json".
        :paramtype content_type: str
        :return: JSON object
        :rtype: JSON
        :raises ~azure.core.exceptions.HttpResponseError:

        Example:
            .. code-block:: python

                # JSON input template you can fill out and use as your body input.
                user_requests_payload = {
                    "operations": [
                        {
                            "payload": {},
                            "type": "str"
                        }
                    ],
                    "comment": "str"
                }

                # response body for status code(s): 200
                response == {
                    "operations": [
                        {
                            "payload": {},
                            "type": "str",
                            "workflowRunIds": [
                                "str"
                            ]
                        }
                    ],
                    "requestId": "str",
                    "requestor": "str",
                    "status": "str",
                    "comment": "str"
                }
        """

    @overload
    async def submit(
        self, user_requests_payload: IO[bytes], *, content_type: str = "application/json", **kwargs: Any
    ) -> JSON:
        """Submit a user request for requestor, a user  request describes user ask to do operation(s) on
        Purview. If any workflow's trigger matches with an operation in request, a run of the workflow
        is created.

        :param user_requests_payload: The payload of submitting a user request. Required.
        :type user_requests_payload: IO[bytes]
        :keyword content_type: Body Parameter content-type. Content type parameter for binary body.
         Default value is "application/json".
        :paramtype content_type: str
        :return: JSON object
        :rtype: JSON
        :raises ~azure.core.exceptions.HttpResponseError:

        Example:
            .. code-block:: python

                # response body for status code(s): 200
                response == {
                    "operations": [
                        {
                            "payload": {},
                            "type": "str",
                            "workflowRunIds": [
                                "str"
                            ]
                        }
                    ],
                    "requestId": "str",
                    "requestor": "str",
                    "status": "str",
                    "comment": "str"
                }
        """

    @distributed_trace_async
    async def submit(self, user_requests_payload: Union[JSON, IO[bytes]], **kwargs: Any) -> JSON:
        """Submit a user request for requestor, a user  request describes user ask to do operation(s) on
        Purview. If any workflow's trigger matches with an operation in request, a run of the workflow
        is created.

        :param user_requests_payload: The payload of submitting a user request. Is either a JSON type
         or a IO[bytes] type. Required.
        :type user_requests_payload: JSON or IO[bytes]
        :return: JSON object
        :rtype: JSON
        :raises ~azure.core.exceptions.HttpResponseError:

        Example:
            .. code-block:: python

                # JSON input template you can fill out and use as your body input.
                user_requests_payload = {
                    "operations": [
                        {
                            "payload": {},
                            "type": "str"
                        }
                    ],
                    "comment": "str"
                }

                # response body for status code(s): 200
                response == {
                    "operations": [
                        {
                            "payload": {},
                            "type": "str",
                            "workflowRunIds": [
                                "str"
                            ]
                        }
                    ],
                    "requestId": "str",
                    "requestor": "str",
                    "status": "str",
                    "comment": "str"
                }
        """
        error_map: MutableMapping = {
            401: ClientAuthenticationError,
            404: ResourceNotFoundError,
            409: ResourceExistsError,
            304: ResourceNotModifiedError,
        }
        error_map.update(kwargs.pop("error_map", {}) or {})

        _headers = case_insensitive_dict(kwargs.pop("headers", {}) or {})
        _params = kwargs.pop("params", {}) or {}

        content_type: Optional[str] = kwargs.pop("content_type", _headers.pop("Content-Type", None))
        cls: ClsType[JSON] = kwargs.pop("cls", None)

        content_type = content_type or "application/json"
        _json = None
        _content = None
        if isinstance(user_requests_payload, (IOBase, bytes)):
            _content = user_requests_payload
        else:
            _json = user_requests_payload

        _request = build_user_requests_submit_request(
            content_type=content_type,
            api_version=self._config.api_version,
            json=_json,
            content=_content,
            headers=_headers,
            params=_params,
        )
        path_format_arguments = {
            "endpoint": self._serialize.url("self._config.endpoint", self._config.endpoint, "str", skip_quote=True),
        }
        _request.url = self._client.format_url(_request.url, **path_format_arguments)

        _stream = False
        pipeline_response: PipelineResponse = await self._client._pipeline.run(  # pylint: disable=protected-access
            _request, stream=_stream, **kwargs
        )

        response = pipeline_response.http_response

        if response.status_code not in [200]:
            map_error(status_code=response.status_code, response=response, error_map=error_map)
            raise HttpResponseError(response=response)

        if response.content:
            deserialized = response.json()
        else:
            deserialized = None

        if cls:
            return cls(pipeline_response, cast(JSON, deserialized), {})  # type: ignore

        return cast(JSON, deserialized)  # type: ignore


class WorkflowRunsOperations:
    """
    .. warning::
        **DO NOT** instantiate this class directly.

        Instead, you should access the following operations through
        :class:`~azure.purview.workflow.aio.PurviewWorkflowClient`'s
        :attr:`workflow_runs` attribute.
    """

    def __init__(self, *args, **kwargs) -> None:
        input_args = list(args)
        self._client = input_args.pop(0) if input_args else kwargs.pop("client")
        self._config = input_args.pop(0) if input_args else kwargs.pop("config")
        self._serialize = input_args.pop(0) if input_args else kwargs.pop("serializer")
        self._deserialize = input_args.pop(0) if input_args else kwargs.pop("deserializer")

    @distributed_trace
    def list(
        self,
        *,
        view_mode: Optional[str] = None,
        time_window: Optional[str] = None,
        orderby: Optional[str] = None,
        run_statuses: Optional[List[str]] = None,
        workflow_ids: Optional[List[str]] = None,
        requestors: Optional[List[str]] = None,
        **kwargs: Any
    ) -> AsyncIterable[JSON]:
        """List workflow runs.

        :keyword view_mode: To filter user's workflow runs or view as admin. Default value is None.
        :paramtype view_mode: str
        :keyword time_window: Time window of filtering items. Known values are: "1d", "7d", "30d", and
         "90d". Default value is None.
        :paramtype time_window: str
        :keyword orderby: The key word which used to sort the results. Known values are: "status desc",
         "status asc", "requestor desc", "requestor asc", "startTime desc", "startTime asc",
         "createdTime desc", and "createdTime asc". Default value is None.
        :paramtype orderby: str
        :keyword run_statuses: Filter workflow runs by workflow run status. Default value is None.
        :paramtype run_statuses: list[str]
        :keyword workflow_ids: Filter items by workflow id list. Default value is None.
        :paramtype workflow_ids: list[str]
        :keyword requestors: Requestors' ids to filter. Default value is None.
        :paramtype requestors: list[str]
        :return: An iterator like instance of JSON object
        :rtype: ~azure.core.async_paging.AsyncItemPaged[JSON]
        :raises ~azure.core.exceptions.HttpResponseError:

        Example:
            .. code-block:: python

                # response body for status code(s): 200
                response == {
                    "id": "str",
                    "requestor": "str",
                    "runPayload": {
                        "targetValue": "str",
                        "type": "str"
                    },
                    "startTime": "2020-02-20 00:00:00",
                    "status": "str",
                    "workflowId": "str",
                    "cancelComment": "str",
                    "cancelTime": "2020-02-20 00:00:00",
                    "endTime": "2020-02-20 00:00:00",
                    "userRequestId": "str"
                }
        """
        _headers = kwargs.pop("headers", {}) or {}
        _params = kwargs.pop("params", {}) or {}

        maxpagesize = kwargs.pop("maxpagesize", None)
        cls: ClsType[JSON] = kwargs.pop("cls", None)

        error_map: MutableMapping = {
            401: ClientAuthenticationError,
            404: ResourceNotFoundError,
            409: ResourceExistsError,
            304: ResourceNotModifiedError,
        }
        error_map.update(kwargs.pop("error_map", {}) or {})

        def prepare_request(next_link=None):
            if not next_link:

                _request = build_workflow_runs_list_request(
                    view_mode=view_mode,
                    time_window=time_window,
                    orderby=orderby,
                    run_statuses=run_statuses,
                    workflow_ids=workflow_ids,
                    requestors=requestors,
                    maxpagesize=maxpagesize,
                    api_version=self._config.api_version,
                    headers=_headers,
                    params=_params,
                )
                path_format_arguments = {
                    "endpoint": self._serialize.url(
                        "self._config.endpoint", self._config.endpoint, "str", skip_quote=True
                    ),
                }
                _request.url = self._client.format_url(_request.url, **path_format_arguments)

            else:
                # make call to next link with the client's api-version
                _parsed_next_link = urllib.parse.urlparse(next_link)
                _next_request_params = case_insensitive_dict(
                    {
                        key: [urllib.parse.quote(v) for v in value]
                        for key, value in urllib.parse.parse_qs(_parsed_next_link.query).items()
                    }
                )
                _next_request_params["api-version"] = self._config.api_version
                _request = HttpRequest(
                    "GET", urllib.parse.urljoin(next_link, _parsed_next_link.path), params=_next_request_params
                )
                path_format_arguments = {
                    "endpoint": self._serialize.url(
                        "self._config.endpoint", self._config.endpoint, "str", skip_quote=True
                    ),
                }
                _request.url = self._client.format_url(_request.url, **path_format_arguments)

            return _request

        async def extract_data(pipeline_response):
            deserialized = pipeline_response.http_response.json()
            list_of_elem = deserialized["value"]
            if cls:
                list_of_elem = cls(list_of_elem)  # type: ignore
            return deserialized.get("nextLink") or None, AsyncList(list_of_elem)

        async def get_next(next_link=None):
            _request = prepare_request(next_link)

            _stream = False
            pipeline_response: PipelineResponse = await self._client._pipeline.run(  # pylint: disable=protected-access
                _request, stream=_stream, **kwargs
            )
            response = pipeline_response.http_response

            if response.status_code not in [200]:
                map_error(status_code=response.status_code, response=response, error_map=error_map)
                raise HttpResponseError(response=response)

            return pipeline_response

        return AsyncItemPaged(get_next, extract_data)


class WorkflowRunOperations:
    """
    .. warning::
        **DO NOT** instantiate this class directly.

        Instead, you should access the following operations through
        :class:`~azure.purview.workflow.aio.PurviewWorkflowClient`'s
        :attr:`workflow_run` attribute.
    """

    def __init__(self, *args, **kwargs) -> None:
        input_args = list(args)
        self._client = input_args.pop(0) if input_args else kwargs.pop("client")
        self._config = input_args.pop(0) if input_args else kwargs.pop("config")
        self._serialize = input_args.pop(0) if input_args else kwargs.pop("serializer")
        self._deserialize = input_args.pop(0) if input_args else kwargs.pop("deserializer")

    @distributed_trace_async
    async def get(self, workflow_run_id: str, **kwargs: Any) -> JSON:
        """Get a workflow run.

        :param workflow_run_id: The workflow run id. Required.
        :type workflow_run_id: str
        :return: JSON object
        :rtype: JSON
        :raises ~azure.core.exceptions.HttpResponseError:

        Example:
            .. code-block:: python

                # response body for status code(s): 200
                response == {
                    "actionDag": {},
                    "detail": {
                        "actions": {},
                        "runInput": {}
                    },
                    "cancelComment": "str",
                    "cancelTime": "2020-02-20 00:00:00",
                    "endTime": "2020-02-20 00:00:00",
                    "id": "str",
                    "requestor": "str",
                    "runPayload": {
                        "payload": {},
                        "targetValue": "str",
                        "type": "str"
                    },
                    "startTime": "2020-02-20 00:00:00",
                    "status": "str",
                    "userRequestId": "str",
                    "workflowId": "str"
                }
        """
        error_map: MutableMapping = {
            401: ClientAuthenticationError,
            404: ResourceNotFoundError,
            409: ResourceExistsError,
            304: ResourceNotModifiedError,
        }
        error_map.update(kwargs.pop("error_map", {}) or {})

        _headers = kwargs.pop("headers", {}) or {}
        _params = kwargs.pop("params", {}) or {}

        cls: ClsType[JSON] = kwargs.pop("cls", None)

        _request = build_workflow_run_get_request(
            workflow_run_id=workflow_run_id,
            api_version=self._config.api_version,
            headers=_headers,
            params=_params,
        )
        path_format_arguments = {
            "endpoint": self._serialize.url("self._config.endpoint", self._config.endpoint, "str", skip_quote=True),
        }
        _request.url = self._client.format_url(_request.url, **path_format_arguments)

        _stream = False
        pipeline_response: PipelineResponse = await self._client._pipeline.run(  # pylint: disable=protected-access
            _request, stream=_stream, **kwargs
        )

        response = pipeline_response.http_response

        if response.status_code not in [200]:
            map_error(status_code=response.status_code, response=response, error_map=error_map)
            raise HttpResponseError(response=response)

        if response.content:
            deserialized = response.json()
        else:
            deserialized = None

        if cls:
            return cls(pipeline_response, cast(JSON, deserialized), {})  # type: ignore

        return cast(JSON, deserialized)  # type: ignore

    @overload
    async def cancel(
        self, workflow_run_id: str, run_cancel_reply: JSON, *, content_type: str = "application/json", **kwargs: Any
    ) -> None:
        """Cancel a workflow run.

        :param workflow_run_id: The workflow run id. Required.
        :type workflow_run_id: str
        :param run_cancel_reply: Reply of canceling a workflow run. Required.
        :type run_cancel_reply: JSON
        :keyword content_type: Body Parameter content-type. Content type parameter for JSON body.
         Default value is "application/json".
        :paramtype content_type: str
        :return: None
        :rtype: None
        :raises ~azure.core.exceptions.HttpResponseError:

        Example:
            .. code-block:: python

                # JSON input template you can fill out and use as your body input.
                run_cancel_reply = {
                    "comment": "str"
                }
        """

    @overload
    async def cancel(
        self,
        workflow_run_id: str,
        run_cancel_reply: IO[bytes],
        *,
        content_type: str = "application/json",
        **kwargs: Any
    ) -> None:
        """Cancel a workflow run.

        :param workflow_run_id: The workflow run id. Required.
        :type workflow_run_id: str
        :param run_cancel_reply: Reply of canceling a workflow run. Required.
        :type run_cancel_reply: IO[bytes]
        :keyword content_type: Body Parameter content-type. Content type parameter for binary body.
         Default value is "application/json".
        :paramtype content_type: str
        :return: None
        :rtype: None
        :raises ~azure.core.exceptions.HttpResponseError:
        """

    @distributed_trace_async
    async def cancel(self, workflow_run_id: str, run_cancel_reply: Union[JSON, IO[bytes]], **kwargs: Any) -> None:
        """Cancel a workflow run.

        :param workflow_run_id: The workflow run id. Required.
        :type workflow_run_id: str
        :param run_cancel_reply: Reply of canceling a workflow run. Is either a JSON type or a
         IO[bytes] type. Required.
        :type run_cancel_reply: JSON or IO[bytes]
        :return: None
        :rtype: None
        :raises ~azure.core.exceptions.HttpResponseError:

        Example:
            .. code-block:: python

                # JSON input template you can fill out and use as your body input.
                run_cancel_reply = {
                    "comment": "str"
                }
        """
        error_map: MutableMapping = {
            401: ClientAuthenticationError,
            404: ResourceNotFoundError,
            409: ResourceExistsError,
            304: ResourceNotModifiedError,
        }
        error_map.update(kwargs.pop("error_map", {}) or {})

        _headers = case_insensitive_dict(kwargs.pop("headers", {}) or {})
        _params = kwargs.pop("params", {}) or {}

        content_type: Optional[str] = kwargs.pop("content_type", _headers.pop("Content-Type", None))
        cls: ClsType[None] = kwargs.pop("cls", None)

        content_type = content_type or "application/json"
        _json = None
        _content = None
        if isinstance(run_cancel_reply, (IOBase, bytes)):
            _content = run_cancel_reply
        else:
            _json = run_cancel_reply

        _request = build_workflow_run_cancel_request(
            workflow_run_id=workflow_run_id,
            content_type=content_type,
            api_version=self._config.api_version,
            json=_json,
            content=_content,
            headers=_headers,
            params=_params,
        )
        path_format_arguments = {
            "endpoint": self._serialize.url("self._config.endpoint", self._config.endpoint, "str", skip_quote=True),
        }
        _request.url = self._client.format_url(_request.url, **path_format_arguments)

        _stream = False
        pipeline_response: PipelineResponse = await self._client._pipeline.run(  # pylint: disable=protected-access
            _request, stream=_stream, **kwargs
        )

        response = pipeline_response.http_response

        if response.status_code not in [200]:
            map_error(status_code=response.status_code, response=response, error_map=error_map)
            raise HttpResponseError(response=response)

        if cls:
            return cls(pipeline_response, None, {})  # type: ignore


class WorkflowTasksOperations:
    """
    .. warning::
        **DO NOT** instantiate this class directly.

        Instead, you should access the following operations through
        :class:`~azure.purview.workflow.aio.PurviewWorkflowClient`'s
        :attr:`workflow_tasks` attribute.
    """

    def __init__(self, *args, **kwargs) -> None:
        input_args = list(args)
        self._client = input_args.pop(0) if input_args else kwargs.pop("client")
        self._config = input_args.pop(0) if input_args else kwargs.pop("config")
        self._serialize = input_args.pop(0) if input_args else kwargs.pop("serializer")
        self._deserialize = input_args.pop(0) if input_args else kwargs.pop("deserializer")

    @distributed_trace
    def list(
        self,
        *,
        view_mode: Optional[str] = None,
        workflow_ids: Optional[List[str]] = None,
        time_window: Optional[str] = None,
        orderby: Optional[str] = None,
        task_types: Optional[List[str]] = None,
        task_statuses: Optional[List[str]] = None,
        requestors: Optional[List[str]] = None,
        assignees: Optional[List[str]] = None,
        workflow_name_keyword: Optional[str] = None,
        **kwargs: Any
    ) -> AsyncIterable[JSON]:
        """Get all workflow tasks.

        :keyword view_mode: To filter user's sent, received or history workflow tasks. Default value is
         None.
        :paramtype view_mode: str
        :keyword workflow_ids: Filter items by workflow id list. Default value is None.
        :paramtype workflow_ids: list[str]
        :keyword time_window: Time window of filtering items. Known values are: "1d", "7d", "30d", and
         "90d". Default value is None.
        :paramtype time_window: str
        :keyword orderby: The key word which used to sort the results. Known values are: "status desc",
         "status asc", "requestor desc", "requestor asc", "startTime desc", "startTime asc",
         "createdTime desc", and "createdTime asc". Default value is None.
        :paramtype orderby: str
        :keyword task_types: Filter items by workflow task type. Default value is None.
        :paramtype task_types: list[str]
        :keyword task_statuses: Filter workflow tasks by status. Default value is None.
        :paramtype task_statuses: list[str]
        :keyword requestors: Requestors' ids to filter. Default value is None.
        :paramtype requestors: list[str]
        :keyword assignees: Assignees' ids to filter. Default value is None.
        :paramtype assignees: list[str]
        :keyword workflow_name_keyword: The key word which could used to filter workflow item with
         related workflow. Default value is None.
        :paramtype workflow_name_keyword: str
        :return: An iterator like instance of JSON object
        :rtype: ~azure.core.async_paging.AsyncItemPaged[JSON]
        :raises ~azure.core.exceptions.HttpResponseError:

        Example:
            .. code-block:: python

                # The response is polymorphic. The following are possible polymorphic responses based
                  off discriminator "type":

                # JSON input template for discriminator value "Approval":
                workflow_task = {
                    "createdTime": "2020-02-20 00:00:00",
                    "id": "str",
                    "lastUpdateTime": "2020-02-20 00:00:00",
                    "payload": {
                        "targetValue": "str",
                        "type": "str",
                        "payload": {}
                    },
                    "requestor": "str",
                    "type": "Approval",
                    "workflowId": "str",
                    "workflowRunId": "str",
                    "approvalDetail": {
                        "approvalType": "str",
                        "approvers": {
                            "str": {
                                "reply": "str",
                                "comment": "str",
                                "responseTime": "2020-02-20 00:00:00"
                            }
                        },
                        "status": "str"
                    },
                    "expiryInfo": {
                        "expirySettings": {
                            "expireAfter": {},
                            "notifyOnExpiration": [
                                "str"
                            ]
                        },
                        "expiryTime": "2020-02-20 00:00:00",
                        "nextExpiryNotificationTime": "2020-02-20 00:00:00",
                        "lastExpiryNotificationTime": "2020-02-20 00:00:00"
                    },
                    "reminderInfo": {
                        "nextRemindTime": "2020-02-20 00:00:00",
                        "reminderSettings": {},
                        "lastRemindTime": "2020-02-20 00:00:00"
                    },
                    "title": "str"
                }

                # JSON input template for discriminator value "SimpleTask":
                workflow_task = {
                    "createdTime": "2020-02-20 00:00:00",
                    "id": "str",
                    "lastUpdateTime": "2020-02-20 00:00:00",
                    "payload": {
                        "targetValue": "str",
                        "type": "str",
                        "payload": {}
                    },
                    "requestor": "str",
                    "type": "SimpleTask",
                    "workflowId": "str",
                    "workflowRunId": "str",
                    "expiryInfo": {
                        "expirySettings": {
                            "expireAfter": {},
                            "notifyOnExpiration": [
                                "str"
                            ]
                        },
                        "expiryTime": "2020-02-20 00:00:00",
                        "nextExpiryNotificationTime": "2020-02-20 00:00:00",
                        "lastExpiryNotificationTime": "2020-02-20 00:00:00"
                    },
                    "reminderInfo": {
                        "nextRemindTime": "2020-02-20 00:00:00",
                        "reminderSettings": {},
                        "lastRemindTime": "2020-02-20 00:00:00"
                    },
                    "taskDetail": {
                        "assignedTo": [
                            "str"
                        ],
                        "changeHistory": [
                            {}
                        ],
                        "status": "str",
                        "taskBody": "str"
                    },
                    "title": "str"
                }

                # response body for status code(s): 200
                response == workflow_task
        """
        _headers = kwargs.pop("headers", {}) or {}
        _params = kwargs.pop("params", {}) or {}

        maxpagesize = kwargs.pop("maxpagesize", None)
        cls: ClsType[JSON] = kwargs.pop("cls", None)

        error_map: MutableMapping = {
            401: ClientAuthenticationError,
            404: ResourceNotFoundError,
            409: ResourceExistsError,
            304: ResourceNotModifiedError,
        }
        error_map.update(kwargs.pop("error_map", {}) or {})

        def prepare_request(next_link=None):
            if not next_link:

                _request = build_workflow_tasks_list_request(
                    view_mode=view_mode,
                    workflow_ids=workflow_ids,
                    time_window=time_window,
                    maxpagesize=maxpagesize,
                    orderby=orderby,
                    task_types=task_types,
                    task_statuses=task_statuses,
                    requestors=requestors,
                    assignees=assignees,
                    workflow_name_keyword=workflow_name_keyword,
                    api_version=self._config.api_version,
                    headers=_headers,
                    params=_params,
                )
                path_format_arguments = {
                    "endpoint": self._serialize.url(
                        "self._config.endpoint", self._config.endpoint, "str", skip_quote=True
                    ),
                }
                _request.url = self._client.format_url(_request.url, **path_format_arguments)

            else:
                # make call to next link with the client's api-version
                _parsed_next_link = urllib.parse.urlparse(next_link)
                _next_request_params = case_insensitive_dict(
                    {
                        key: [urllib.parse.quote(v) for v in value]
                        for key, value in urllib.parse.parse_qs(_parsed_next_link.query).items()
                    }
                )
                _next_request_params["api-version"] = self._config.api_version
                _request = HttpRequest(
                    "GET", urllib.parse.urljoin(next_link, _parsed_next_link.path), params=_next_request_params
                )
                path_format_arguments = {
                    "endpoint": self._serialize.url(
                        "self._config.endpoint", self._config.endpoint, "str", skip_quote=True
                    ),
                }
                _request.url = self._client.format_url(_request.url, **path_format_arguments)

            return _request

        async def extract_data(pipeline_response):
            deserialized = pipeline_response.http_response.json()
            list_of_elem = deserialized["value"]
            if cls:
                list_of_elem = cls(list_of_elem)  # type: ignore
            return deserialized.get("nextLink") or None, AsyncList(list_of_elem)

        async def get_next(next_link=None):
            _request = prepare_request(next_link)

            _stream = False
            pipeline_response: PipelineResponse = await self._client._pipeline.run(  # pylint: disable=protected-access
                _request, stream=_stream, **kwargs
            )
            response = pipeline_response.http_response

            if response.status_code not in [200]:
                map_error(status_code=response.status_code, response=response, error_map=error_map)
                raise HttpResponseError(response=response)

            return pipeline_response

        return AsyncItemPaged(get_next, extract_data)


class WorkflowTaskOperations:
    """
    .. warning::
        **DO NOT** instantiate this class directly.

        Instead, you should access the following operations through
        :class:`~azure.purview.workflow.aio.PurviewWorkflowClient`'s
        :attr:`workflow_task` attribute.
    """

    def __init__(self, *args, **kwargs) -> None:
        input_args = list(args)
        self._client = input_args.pop(0) if input_args else kwargs.pop("client")
        self._config = input_args.pop(0) if input_args else kwargs.pop("config")
        self._serialize = input_args.pop(0) if input_args else kwargs.pop("serializer")
        self._deserialize = input_args.pop(0) if input_args else kwargs.pop("deserializer")

    @distributed_trace_async
    async def get(self, task_id: str, **kwargs: Any) -> JSON:
        """Get a workflow task.

        :param task_id: The task id. Required.
        :type task_id: str
        :return: JSON object
        :rtype: JSON
        :raises ~azure.core.exceptions.HttpResponseError:

        Example:
            .. code-block:: python

                # The response is polymorphic. The following are possible polymorphic responses based
                  off discriminator "type":

                # JSON input template for discriminator value "Approval":
                workflow_task = {
                    "createdTime": "2020-02-20 00:00:00",
                    "id": "str",
                    "lastUpdateTime": "2020-02-20 00:00:00",
                    "payload": {
                        "targetValue": "str",
                        "type": "str",
                        "payload": {}
                    },
                    "requestor": "str",
                    "type": "Approval",
                    "workflowId": "str",
                    "workflowRunId": "str",
                    "approvalDetail": {
                        "approvalType": "str",
                        "approvers": {
                            "str": {
                                "reply": "str",
                                "comment": "str",
                                "responseTime": "2020-02-20 00:00:00"
                            }
                        },
                        "status": "str"
                    },
                    "expiryInfo": {
                        "expirySettings": {
                            "expireAfter": {},
                            "notifyOnExpiration": [
                                "str"
                            ]
                        },
                        "expiryTime": "2020-02-20 00:00:00",
                        "nextExpiryNotificationTime": "2020-02-20 00:00:00",
                        "lastExpiryNotificationTime": "2020-02-20 00:00:00"
                    },
                    "reminderInfo": {
                        "nextRemindTime": "2020-02-20 00:00:00",
                        "reminderSettings": {},
                        "lastRemindTime": "2020-02-20 00:00:00"
                    },
                    "title": "str"
                }

                # JSON input template for discriminator value "SimpleTask":
                workflow_task = {
                    "createdTime": "2020-02-20 00:00:00",
                    "id": "str",
                    "lastUpdateTime": "2020-02-20 00:00:00",
                    "payload": {
                        "targetValue": "str",
                        "type": "str",
                        "payload": {}
                    },
                    "requestor": "str",
                    "type": "SimpleTask",
                    "workflowId": "str",
                    "workflowRunId": "str",
                    "expiryInfo": {
                        "expirySettings": {
                            "expireAfter": {},
                            "notifyOnExpiration": [
                                "str"
                            ]
                        },
                        "expiryTime": "2020-02-20 00:00:00",
                        "nextExpiryNotificationTime": "2020-02-20 00:00:00",
                        "lastExpiryNotificationTime": "2020-02-20 00:00:00"
                    },
                    "reminderInfo": {
                        "nextRemindTime": "2020-02-20 00:00:00",
                        "reminderSettings": {},
                        "lastRemindTime": "2020-02-20 00:00:00"
                    },
                    "taskDetail": {
                        "assignedTo": [
                            "str"
                        ],
                        "changeHistory": [
                            {}
                        ],
                        "status": "str",
                        "taskBody": "str"
                    },
                    "title": "str"
                }

                # response body for status code(s): 200
                response == workflow_task
        """
        error_map: MutableMapping = {
            401: ClientAuthenticationError,
            404: ResourceNotFoundError,
            409: ResourceExistsError,
            304: ResourceNotModifiedError,
        }
        error_map.update(kwargs.pop("error_map", {}) or {})

        _headers = kwargs.pop("headers", {}) or {}
        _params = kwargs.pop("params", {}) or {}

        cls: ClsType[JSON] = kwargs.pop("cls", None)

        _request = build_workflow_task_get_request(
            task_id=task_id,
            headers=_headers,
            params=_params,
        )
        path_format_arguments = {
            "endpoint": self._serialize.url("self._config.endpoint", self._config.endpoint, "str", skip_quote=True),
        }
        _request.url = self._client.format_url(_request.url, **path_format_arguments)

        _stream = False
        pipeline_response: PipelineResponse = await self._client._pipeline.run(  # pylint: disable=protected-access
            _request, stream=_stream, **kwargs
        )

        response = pipeline_response.http_response

        if response.status_code not in [200]:
            map_error(status_code=response.status_code, response=response, error_map=error_map)
            raise HttpResponseError(response=response)

        if response.content:
            deserialized = response.json()
        else:
            deserialized = None

        if cls:
            return cls(pipeline_response, cast(JSON, deserialized), {})  # type: ignore

        return cast(JSON, deserialized)  # type: ignore

    @overload
    async def reassign(
        self, task_id: str, reassign_command: JSON, *, content_type: str = "application/json", **kwargs: Any
    ) -> None:
        """Reassign a workflow task.

        :param task_id: The task id. Required.
        :type task_id: str
        :param reassign_command: The request body of reassigning a workflow task. Required.
        :type reassign_command: JSON
        :keyword content_type: Body Parameter content-type. Content type parameter for JSON body.
         Default value is "application/json".
        :paramtype content_type: str
        :return: None
        :rtype: None
        :raises ~azure.core.exceptions.HttpResponseError:

        Example:
            .. code-block:: python

                # JSON input template you can fill out and use as your body input.
                reassign_command = {
                    "reassignments": [
                        {
                            "reassignFrom": "str",
                            "reassignTo": "str"
                        }
                    ]
                }
        """

    @overload
    async def reassign(
        self, task_id: str, reassign_command: IO[bytes], *, content_type: str = "application/json", **kwargs: Any
    ) -> None:
        """Reassign a workflow task.

        :param task_id: The task id. Required.
        :type task_id: str
        :param reassign_command: The request body of reassigning a workflow task. Required.
        :type reassign_command: IO[bytes]
        :keyword content_type: Body Parameter content-type. Content type parameter for binary body.
         Default value is "application/json".
        :paramtype content_type: str
        :return: None
        :rtype: None
        :raises ~azure.core.exceptions.HttpResponseError:
        """

    @distributed_trace_async
    async def reassign(self, task_id: str, reassign_command: Union[JSON, IO[bytes]], **kwargs: Any) -> None:
        """Reassign a workflow task.

        :param task_id: The task id. Required.
        :type task_id: str
        :param reassign_command: The request body of reassigning a workflow task. Is either a JSON type
         or a IO[bytes] type. Required.
        :type reassign_command: JSON or IO[bytes]
        :return: None
        :rtype: None
        :raises ~azure.core.exceptions.HttpResponseError:

        Example:
            .. code-block:: python

                # JSON input template you can fill out and use as your body input.
                reassign_command = {
                    "reassignments": [
                        {
                            "reassignFrom": "str",
                            "reassignTo": "str"
                        }
                    ]
                }
        """
        error_map: MutableMapping = {
            401: ClientAuthenticationError,
            404: ResourceNotFoundError,
            409: ResourceExistsError,
            304: ResourceNotModifiedError,
        }
        error_map.update(kwargs.pop("error_map", {}) or {})

        _headers = case_insensitive_dict(kwargs.pop("headers", {}) or {})
        _params = kwargs.pop("params", {}) or {}

        content_type: Optional[str] = kwargs.pop("content_type", _headers.pop("Content-Type", None))
        cls: ClsType[None] = kwargs.pop("cls", None)

        content_type = content_type or "application/json"
        _json = None
        _content = None
        if isinstance(reassign_command, (IOBase, bytes)):
            _content = reassign_command
        else:
            _json = reassign_command

        _request = build_workflow_task_reassign_request(
            task_id=task_id,
            content_type=content_type,
            api_version=self._config.api_version,
            json=_json,
            content=_content,
            headers=_headers,
            params=_params,
        )
        path_format_arguments = {
            "endpoint": self._serialize.url("self._config.endpoint", self._config.endpoint, "str", skip_quote=True),
        }
        _request.url = self._client.format_url(_request.url, **path_format_arguments)

        _stream = False
        pipeline_response: PipelineResponse = await self._client._pipeline.run(  # pylint: disable=protected-access
            _request, stream=_stream, **kwargs
        )

        response = pipeline_response.http_response

        if response.status_code not in [200]:
            map_error(status_code=response.status_code, response=response, error_map=error_map)
            raise HttpResponseError(response=response)

        if cls:
            return cls(pipeline_response, None, {})  # type: ignore


class ApprovalOperations:
    """
    .. warning::
        **DO NOT** instantiate this class directly.

        Instead, you should access the following operations through
        :class:`~azure.purview.workflow.aio.PurviewWorkflowClient`'s
        :attr:`approval` attribute.
    """

    def __init__(self, *args, **kwargs) -> None:
        input_args = list(args)
        self._client = input_args.pop(0) if input_args else kwargs.pop("client")
        self._config = input_args.pop(0) if input_args else kwargs.pop("config")
        self._serialize = input_args.pop(0) if input_args else kwargs.pop("serializer")
        self._deserialize = input_args.pop(0) if input_args else kwargs.pop("deserializer")

    @overload
    async def approve(
        self, task_id: str, approval_response_comment: JSON, *, content_type: str = "application/json", **kwargs: Any
    ) -> None:
        """Approve an approval.

        :param task_id: The task id. Required.
        :type task_id: str
        :param approval_response_comment: The request body of approving an approval type of workflow
         task. Required.
        :type approval_response_comment: JSON
        :keyword content_type: Body Parameter content-type. Content type parameter for JSON body.
         Default value is "application/json".
        :paramtype content_type: str
        :return: None
        :rtype: None
        :raises ~azure.core.exceptions.HttpResponseError:

        Example:
            .. code-block:: python

                # JSON input template you can fill out and use as your body input.
                approval_response_comment = {
                    "comment": "str"
                }
        """

    @overload
    async def approve(
        self,
        task_id: str,
        approval_response_comment: IO[bytes],
        *,
        content_type: str = "application/json",
        **kwargs: Any
    ) -> None:
        """Approve an approval.

        :param task_id: The task id. Required.
        :type task_id: str
        :param approval_response_comment: The request body of approving an approval type of workflow
         task. Required.
        :type approval_response_comment: IO[bytes]
        :keyword content_type: Body Parameter content-type. Content type parameter for binary body.
         Default value is "application/json".
        :paramtype content_type: str
        :return: None
        :rtype: None
        :raises ~azure.core.exceptions.HttpResponseError:
        """

    @distributed_trace_async
    async def approve(self, task_id: str, approval_response_comment: Union[JSON, IO[bytes]], **kwargs: Any) -> None:
        """Approve an approval.

        :param task_id: The task id. Required.
        :type task_id: str
        :param approval_response_comment: The request body of approving an approval type of workflow
         task. Is either a JSON type or a IO[bytes] type. Required.
        :type approval_response_comment: JSON or IO[bytes]
        :return: None
        :rtype: None
        :raises ~azure.core.exceptions.HttpResponseError:

        Example:
            .. code-block:: python

                # JSON input template you can fill out and use as your body input.
                approval_response_comment = {
                    "comment": "str"
                }
        """
        error_map: MutableMapping = {
            401: ClientAuthenticationError,
            404: ResourceNotFoundError,
            409: ResourceExistsError,
            304: ResourceNotModifiedError,
        }
        error_map.update(kwargs.pop("error_map", {}) or {})

        _headers = case_insensitive_dict(kwargs.pop("headers", {}) or {})
        _params = kwargs.pop("params", {}) or {}

        content_type: Optional[str] = kwargs.pop("content_type", _headers.pop("Content-Type", None))
        cls: ClsType[None] = kwargs.pop("cls", None)

        content_type = content_type or "application/json"
        _json = None
        _content = None
        if isinstance(approval_response_comment, (IOBase, bytes)):
            _content = approval_response_comment
        else:
            _json = approval_response_comment

        _request = build_approval_approve_request(
            task_id=task_id,
            content_type=content_type,
            api_version=self._config.api_version,
            json=_json,
            content=_content,
            headers=_headers,
            params=_params,
        )
        path_format_arguments = {
            "endpoint": self._serialize.url("self._config.endpoint", self._config.endpoint, "str", skip_quote=True),
        }
        _request.url = self._client.format_url(_request.url, **path_format_arguments)

        _stream = False
        pipeline_response: PipelineResponse = await self._client._pipeline.run(  # pylint: disable=protected-access
            _request, stream=_stream, **kwargs
        )

        response = pipeline_response.http_response

        if response.status_code not in [200]:
            map_error(status_code=response.status_code, response=response, error_map=error_map)
            raise HttpResponseError(response=response)

        if cls:
            return cls(pipeline_response, None, {})  # type: ignore

    @overload
    async def reject(
        self, task_id: str, approval_response_comment: JSON, *, content_type: str = "application/json", **kwargs: Any
    ) -> None:
        """Reject an approval.

        :param task_id: The task id. Required.
        :type task_id: str
        :param approval_response_comment: The request body of rejecting an approval type of workflow
         task. Required.
        :type approval_response_comment: JSON
        :keyword content_type: Body Parameter content-type. Content type parameter for JSON body.
         Default value is "application/json".
        :paramtype content_type: str
        :return: None
        :rtype: None
        :raises ~azure.core.exceptions.HttpResponseError:

        Example:
            .. code-block:: python

                # JSON input template you can fill out and use as your body input.
                approval_response_comment = {
                    "comment": "str"
                }
        """

    @overload
    async def reject(
        self,
        task_id: str,
        approval_response_comment: IO[bytes],
        *,
        content_type: str = "application/json",
        **kwargs: Any
    ) -> None:
        """Reject an approval.

        :param task_id: The task id. Required.
        :type task_id: str
        :param approval_response_comment: The request body of rejecting an approval type of workflow
         task. Required.
        :type approval_response_comment: IO[bytes]
        :keyword content_type: Body Parameter content-type. Content type parameter for binary body.
         Default value is "application/json".
        :paramtype content_type: str
        :return: None
        :rtype: None
        :raises ~azure.core.exceptions.HttpResponseError:
        """

    @distributed_trace_async
    async def reject(self, task_id: str, approval_response_comment: Union[JSON, IO[bytes]], **kwargs: Any) -> None:
        """Reject an approval.

        :param task_id: The task id. Required.
        :type task_id: str
        :param approval_response_comment: The request body of rejecting an approval type of workflow
         task. Is either a JSON type or a IO[bytes] type. Required.
        :type approval_response_comment: JSON or IO[bytes]
        :return: None
        :rtype: None
        :raises ~azure.core.exceptions.HttpResponseError:

        Example:
            .. code-block:: python

                # JSON input template you can fill out and use as your body input.
                approval_response_comment = {
                    "comment": "str"
                }
        """
        error_map: MutableMapping = {
            401: ClientAuthenticationError,
            404: ResourceNotFoundError,
            409: ResourceExistsError,
            304: ResourceNotModifiedError,
        }
        error_map.update(kwargs.pop("error_map", {}) or {})

        _headers = case_insensitive_dict(kwargs.pop("headers", {}) or {})
        _params = kwargs.pop("params", {}) or {}

        content_type: Optional[str] = kwargs.pop("content_type", _headers.pop("Content-Type", None))
        cls: ClsType[None] = kwargs.pop("cls", None)

        content_type = content_type or "application/json"
        _json = None
        _content = None
        if isinstance(approval_response_comment, (IOBase, bytes)):
            _content = approval_response_comment
        else:
            _json = approval_response_comment

        _request = build_approval_reject_request(
            task_id=task_id,
            content_type=content_type,
            api_version=self._config.api_version,
            json=_json,
            content=_content,
            headers=_headers,
            params=_params,
        )
        path_format_arguments = {
            "endpoint": self._serialize.url("self._config.endpoint", self._config.endpoint, "str", skip_quote=True),
        }
        _request.url = self._client.format_url(_request.url, **path_format_arguments)

        _stream = False
        pipeline_response: PipelineResponse = await self._client._pipeline.run(  # pylint: disable=protected-access
            _request, stream=_stream, **kwargs
        )

        response = pipeline_response.http_response

        if response.status_code not in [200]:
            map_error(status_code=response.status_code, response=response, error_map=error_map)
            raise HttpResponseError(response=response)

        if cls:
            return cls(pipeline_response, None, {})  # type: ignore


class TaskStatusOperations:
    """
    .. warning::
        **DO NOT** instantiate this class directly.

        Instead, you should access the following operations through
        :class:`~azure.purview.workflow.aio.PurviewWorkflowClient`'s
        :attr:`task_status` attribute.
    """

    def __init__(self, *args, **kwargs) -> None:
        input_args = list(args)
        self._client = input_args.pop(0) if input_args else kwargs.pop("client")
        self._config = input_args.pop(0) if input_args else kwargs.pop("config")
        self._serialize = input_args.pop(0) if input_args else kwargs.pop("serializer")
        self._deserialize = input_args.pop(0) if input_args else kwargs.pop("deserializer")

    @overload
    async def update(
        self, task_id: str, task_update_command: JSON, *, content_type: str = "application/json", **kwargs: Any
    ) -> None:
        """Update the status of a workflow task request.

        :param task_id: The task id. Required.
        :type task_id: str
        :param task_update_command: Request body of updating workflow task request. Required.
        :type task_update_command: JSON
        :keyword content_type: Body Parameter content-type. Content type parameter for JSON body.
         Default value is "application/json".
        :paramtype content_type: str
        :return: None
        :rtype: None
        :raises ~azure.core.exceptions.HttpResponseError:

        Example:
            .. code-block:: python

                # JSON input template you can fill out and use as your body input.
                task_update_command = {
                    "newStatus": "str",
                    "comment": "str"
                }
        """

    @overload
    async def update(
        self, task_id: str, task_update_command: IO[bytes], *, content_type: str = "application/json", **kwargs: Any
    ) -> None:
        """Update the status of a workflow task request.

        :param task_id: The task id. Required.
        :type task_id: str
        :param task_update_command: Request body of updating workflow task request. Required.
        :type task_update_command: IO[bytes]
        :keyword content_type: Body Parameter content-type. Content type parameter for binary body.
         Default value is "application/json".
        :paramtype content_type: str
        :return: None
        :rtype: None
        :raises ~azure.core.exceptions.HttpResponseError:
        """

    @distributed_trace_async
    async def update(self, task_id: str, task_update_command: Union[JSON, IO[bytes]], **kwargs: Any) -> None:
        """Update the status of a workflow task request.

        :param task_id: The task id. Required.
        :type task_id: str
        :param task_update_command: Request body of updating workflow task request. Is either a JSON
         type or a IO[bytes] type. Required.
        :type task_update_command: JSON or IO[bytes]
        :return: None
        :rtype: None
        :raises ~azure.core.exceptions.HttpResponseError:

        Example:
            .. code-block:: python

                # JSON input template you can fill out and use as your body input.
                task_update_command = {
                    "newStatus": "str",
                    "comment": "str"
                }
        """
        error_map: MutableMapping = {
            401: ClientAuthenticationError,
            404: ResourceNotFoundError,
            409: ResourceExistsError,
            304: ResourceNotModifiedError,
        }
        error_map.update(kwargs.pop("error_map", {}) or {})

        _headers = case_insensitive_dict(kwargs.pop("headers", {}) or {})
        _params = kwargs.pop("params", {}) or {}

        content_type: Optional[str] = kwargs.pop("content_type", _headers.pop("Content-Type", None))
        cls: ClsType[None] = kwargs.pop("cls", None)

        content_type = content_type or "application/json"
        _json = None
        _content = None
        if isinstance(task_update_command, (IOBase, bytes)):
            _content = task_update_command
        else:
            _json = task_update_command

        _request = build_task_status_update_request(
            task_id=task_id,
            content_type=content_type,
            api_version=self._config.api_version,
            json=_json,
            content=_content,
            headers=_headers,
            params=_params,
        )
        path_format_arguments = {
            "endpoint": self._serialize.url("self._config.endpoint", self._config.endpoint, "str", skip_quote=True),
        }
        _request.url = self._client.format_url(_request.url, **path_format_arguments)

        _stream = False
        pipeline_response: PipelineResponse = await self._client._pipeline.run(  # pylint: disable=protected-access
            _request, stream=_stream, **kwargs
        )

        response = pipeline_response.http_response

        if response.status_code not in [200]:
            map_error(status_code=response.status_code, response=response, error_map=error_map)
            raise HttpResponseError(response=response)

        if cls:
            return cls(pipeline_response, None, {})  # type: ignore
