/**
 * Copyright (C) 2006-2009 Dustin Sallings
 * Copyright (C) 2009-2011 Couchbase, Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALING
 * IN THE SOFTWARE.
 */

package net.spy.memcached;

import java.net.InetSocketAddress;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.jmock.Mock;

/**
 * Test ketama node location.
 */
public class KetamaNodeLocatorTest extends AbstractNodeLocationCase {

  protected void setupNodes(HashAlgorithm alg, int n) {
    super.setupNodes(n);
    for (int i = 0; i < nodeMocks.length; i++) {
      nodeMocks[i].expects(atLeastOnce()).method("getSocketAddress")
          .will(returnValue(InetSocketAddress.createUnresolved("127.0.0.1",
          10000 + i)));
    }

    locator = new KetamaNodeLocator(Arrays.asList(nodes), alg);
  }

  @Override
  protected void setupNodes(int n) {
    setupNodes(DefaultHashAlgorithm.KETAMA_HASH, n);
  }

  public void testAll() throws Exception {
    setupNodes(4);

    Collection<MemcachedNode> all = locator.getAll();
    assertEquals(4, all.size());
    for (int i = 0; i < 4; i++) {
      assertTrue(all.contains(nodes[i]));
    }
  }

  public void testAllAfterUpdate() throws Exception {
    setupNodes(4);

    Collection<MemcachedNode> all = locator.getAll();
    assertEquals(4, all.size());
    for (int i = 0; i < 4; i++) {
      assertTrue(all.contains(nodes[i]));
    }

    ArrayList<MemcachedNode> toUpdate = new ArrayList<MemcachedNode>();
    Mock mock = mock(MemcachedNode.class);
    mock.expects(atLeastOnce()).method("getSocketAddress")
          .will(returnValue(InetSocketAddress.createUnresolved("127.0.0.1",
          10000)));
    toUpdate.add((MemcachedNode) mock.proxy());
    locator.updateLocator(toUpdate);

    Collection<MemcachedNode> afterUpdate = locator.getAll();
    assertEquals(1, afterUpdate.size());
    for (int i = 0; i < 4; i++) {
      assertFalse(afterUpdate.contains(nodes[i]));
    }
    assertTrue(afterUpdate.contains((MemcachedNode) mock.proxy()));
  }

  public void testAllClone() throws Exception {
    setupNodes(4);

    Collection<MemcachedNode> all = locator.getReadonlyCopy().getAll();
    assertEquals(4, all.size());
  }

  public void testLookups() {
    setupNodes(4);
    assertSame(nodes[0], locator.getPrimary("dustin"));
    assertSame(nodes[2], locator.getPrimary("noelani"));
    assertSame(nodes[0], locator.getPrimary("some other key"));
  }

  public void testLookupsClone() {
    setupNodes(4);
    assertSame(nodes[0].toString(),
        locator.getReadonlyCopy().getPrimary("dustin").toString());
    assertSame(nodes[2].toString(),
        locator.getReadonlyCopy().getPrimary("noelani").toString());
    assertSame(nodes[0].toString(),
        locator.getReadonlyCopy().getPrimary("some other key").toString());
  }

  public void testContinuumWrapping() {
    setupNodes(4);
    // This is the method by which I found something that would wrap
    /*
     * String key="a"; // maximum key found in the ketama continuum long
     * lastKey=((KetamaNodeLocator)locator).getMaxKey();
     * while(HashAlgorithm.KETAMA_HASH.hash(key) < lastKey) {
     * key=PwGen.getPass(8); }
     * System.out.println("Found a key past the end of the continuum:  " + key);
     */
    assertEquals(4294887009L, ((KetamaNodeLocator) locator).getMaxKey());

    assertSame(nodes[3], locator.getPrimary("V5XS8C8N"));
    assertSame(nodes[3], locator.getPrimary("8KR2DKR2"));
    assertSame(nodes[3], locator.getPrimary("L9KH6X4X"));
  }

  public void testClusterResizing() {
    setupNodes(4);
    assertSame(nodes[0], locator.getPrimary("dustin"));
    assertSame(nodes[2], locator.getPrimary("noelani"));
    assertSame(nodes[0], locator.getPrimary("some other key"));

    setupNodes(5);
    assertSame(nodes[0], locator.getPrimary("dustin"));
    assertSame(nodes[2], locator.getPrimary("noelani"));
    assertSame(nodes[4], locator.getPrimary("some other key"));
  }

  public void testSequence1() {
    setupNodes(4);
    assertSequence("dustin", 0, 2, 1, 2, 3, 2, 0);
  }

  public void testSequence2() {
    setupNodes(4);
    assertSequence("noelani", 2, 1, 1, 3, 2, 2, 3);
  }

  private void assertPosForKey(String k, int nid) {
    assertSame(nodes[nid], locator.getPrimary(k));
  }

  public void testLibKetamaCompat() {
    setupNodes(5);
    assertPosForKey("36", 2);
    assertPosForKey("10037", 3);
    assertPosForKey("22051", 1);
    assertPosForKey("49044", 4);
  }

  public void testFNV1A32() {
    HashAlgorithm alg = DefaultHashAlgorithm.FNV1A_32_HASH;
    setupNodes(alg, 5);
    assertSequence("noelani", 1, 2, 2, 2, 3, 4, 2);

    assertSame(nodes[2], locator.getPrimary("dustin"));
    assertSame(nodes[1], locator.getPrimary("noelani"));
    assertSame(nodes[4], locator.getPrimary("some other key"));
  }

  private MemcachedNode[] mockNodes(String[] servers) {
    setupNodes(servers.length);

    for (int i = 0; i < nodeMocks.length; i++) {
      List<InetSocketAddress> a = AddrUtil.getAddresses(servers[i]);

      nodeMocks[i].expects(atLeastOnce()).method("getSocketAddress")
          .will(returnValue(a.iterator().next()));

    }
    return nodes;
  }


  public void testLibKetamaWeightedCompatLibmemcached() {
      String[] servers = {"127.0.0.1:11211", "127.0.0.1:11212"};
      List<MemcachedNode> nodes = Arrays.asList(mockNodes(servers));
      Map<InetSocketAddress, Integer> weights = new HashMap<InetSocketAddress, Integer>();
      weights.put((InetSocketAddress) nodes.get(0).getSocketAddress(), 8);
      weights.put((InetSocketAddress) nodes.get(1).getSocketAddress(), 8);

      locator = new KetamaNodeLocator(nodes, DefaultHashAlgorithm.KETAMA_HASH,
              KetamaNodeKeyFormatter.Format.LIBMEMCACHED, weights);

      String[][] exp = { { "key1", "localhost/127.0.0.1:11212" }, { "key2", "localhost/127.0.0.1:11211" },
              { "key3", "localhost/127.0.0.1:11212" }, { "key4", "localhost/127.0.0.1:11212" } };

      for (String[] s : exp) {
          String k = s[0];
          String server = s[1];
          MemcachedNode n = locator.getPrimary(k);
          assertEquals(server, n.getSocketAddress().toString());
      }
  }

  public void testLibKetamaCompatTwo() {
    String[] servers = {"10.0.1.1:11211", "10.0.1.2:11211",
      "10.0.1.3:11211", "10.0.1.4:11211", "10.0.1.5:11211",
      "10.0.1.6:11211", "10.0.1.7:11211", "10.0.1.8:11211"};
    locator = new KetamaNodeLocator(Arrays.asList(mockNodes(servers)),
        DefaultHashAlgorithm.KETAMA_HASH);

    String[][] exp = { { "137470", "10.0.1.2:11211" }, { "137703", "10.0.1.3:11211" },
      { "137936", "10.0.1.1:11211" }, { "138169", "10.0.1.6:11211" },
      { "138402", "10.0.1.8:11211" }, { "138635", "10.0.1.7:11211" },
      { "138868", "10.0.1.2:11211" }, { "139101", "10.0.1.4:11211" },
      { "139334", "10.0.1.7:11211" }, { "139567", "10.0.1.6:11211" },
      { "139800", "10.0.1.1:11211" }, { "140033", "10.0.1.8:11211" },
      { "140266", "10.0.1.7:11211" }, { "140499", "10.0.1.3:11211" },
      { "140732", "10.0.1.2:11211" }, { "140965", "10.0.1.7:11211" },
      { "141198", "10.0.1.6:11211" }, { "141431", "10.0.1.7:11211" },
      { "141664", "10.0.1.5:11211" }, { "141897", "10.0.1.5:11211" },
      { "142130", "10.0.1.5:11211" }, { "142363", "10.0.1.4:11211" },
      { "142596", "10.0.1.8:11211" }, { "142829", "10.0.1.2:11211" },
      { "143062", "10.0.1.2:11211" }, { "143295", "10.0.1.4:11211" },
      { "143528", "10.0.1.8:11211" }, { "143761", "10.0.1.2:11211" },
      { "143994", "10.0.1.5:11211" }, { "144227", "10.0.1.3:11211" },
      { "144460", "10.0.1.2:11211" }, { "152149", "10.0.1.2:11211" },
      { "152382", "10.0.1.2:11211" }, { "152615", "10.0.1.3:11211" },
      { "152848", "10.0.1.5:11211" }, { "153081", "10.0.1.1:11211" },
      { "153314", "10.0.1.1:11211" }, { "153547", "10.0.1.1:11211" },
      { "153780", "10.0.1.1:11211" }, { "154013", "10.0.1.2:11211" },
      { "154246", "10.0.1.1:11211" }, { "154479", "10.0.1.8:11211" },
      { "154712", "10.0.1.1:11211" }, { "154945", "10.0.1.1:11211" },
      { "155178", "10.0.1.8:11211" }, { "155411", "10.0.1.8:11211" },
      { "155644", "10.0.1.8:11211" }, { "155877", "10.0.1.2:11211" },
      { "156110", "10.0.1.3:11211" }, { "156343", "10.0.1.1:11211" },
      { "156576", "10.0.1.7:11211" }, { "156809", "10.0.1.7:11211" },
      { "157042", "10.0.1.1:11211" }, { "157275", "10.0.1.7:11211" },
      { "157508", "10.0.1.2:11211" }, { "157741", "10.0.1.7:11211" },
      { "157974", "10.0.1.5:11211" }, { "158207", "10.0.1.5:11211" },
      { "158440", "10.0.1.4:11211" }, { "158673", "10.0.1.3:11211" },
      { "158906", "10.0.1.3:11211" }, { "159139", "10.0.1.8:11211" },
      { "159372", "10.0.1.6:11211" }, { "159605", "10.0.1.3:11211" },
      { "159838", "10.0.1.4:11211" }, { "160071", "10.0.1.2:11211" },
      { "160304", "10.0.1.4:11211" }, { "160537", "10.0.1.6:11211" },
      { "160770", "10.0.1.5:11211" }, { "161003", "10.0.1.3:11211" },
      { "161236", "10.0.1.7:11211" }, { "161469", "10.0.1.5:11211" },
      { "161702", "10.0.1.7:11211" }, { "161935", "10.0.1.8:11211" },
      { "162168", "10.0.1.8:11211" }, { "162401", "10.0.1.8:11211" },
      { "162634", "10.0.1.8:11211" }, { "162867", "10.0.1.6:11211" },
      { "163100", "10.0.1.8:11211" }, { "163333", "10.0.1.7:11211" },
      { "163566", "10.0.1.7:11211" }, { "163799", "10.0.1.3:11211" },
      { "164032", "10.0.1.6:11211" }, { "164265", "10.0.1.8:11211" },
      { "169158", "10.0.1.7:11211" }, { "169391", "10.0.1.6:11211" },
      { "169624", "10.0.1.6:11211" }, { "169857", "10.0.1.6:11211" },
      { "170090", "10.0.1.8:11211" }, { "170323", "10.0.1.4:11211" },
      { "170556", "10.0.1.2:11211" }, { "170789", "10.0.1.8:11211" },
      { "171022", "10.0.1.8:11211" }, { "171255", "10.0.1.4:11211" },
      { "171488", "10.0.1.7:11211" }, { "171721", "10.0.1.4:11211" },
      { "171954", "10.0.1.5:11211" }, { "172187", "10.0.1.4:11211" },
      { "172420", "10.0.1.3:11211" }, { "172653", "10.0.1.2:11211" },
      { "172886", "10.0.1.1:11211" }, { "173119", "10.0.1.8:11211" },
      { "173352", "10.0.1.5:11211" }, { "176614", "10.0.1.3:11211" },
      { "176847", "10.0.1.8:11211" }, { "177080", "10.0.1.1:11211" },
      { "177313", "10.0.1.4:11211" }, { "177546", "10.0.1.2:11211" },
      { "177779", "10.0.1.8:11211" }, { "178012", "10.0.1.6:11211" },
      { "178245", "10.0.1.3:11211" }, { "178478", "10.0.1.7:11211" },
      { "178711", "10.0.1.3:11211" }, { "178944", "10.0.1.8:11211" },
      { "179177", "10.0.1.1:11211" }, { "179410", "10.0.1.6:11211" },
      { "179643", "10.0.1.1:11211" }, { "179876", "10.0.1.8:11211" },
      { "180109", "10.0.1.3:11211" }, { "180342", "10.0.1.2:11211" },
      { "180575", "10.0.1.7:11211" }, { "180808", "10.0.1.8:11211" },
      { "181041", "10.0.1.5:11211" }, { "181274", "10.0.1.6:11211" },
      { "181507", "10.0.1.8:11211" }, { "181740", "10.0.1.6:11211" },
      { "181973", "10.0.1.5:11211" }, { "182206", "10.0.1.4:11211" },
      { "182439", "10.0.1.2:11211" }, { "182672", "10.0.1.8:11211" },
      { "182905", "10.0.1.8:11211" }, { "183138", "10.0.1.4:11211" },
      { "183371", "10.0.1.8:11211" }, { "183604", "10.0.1.7:11211" },
      { "183837", "10.0.1.7:11211" }, { "184070", "10.0.1.4:11211" },
      { "184303", "10.0.1.7:11211" }, { "184536", "10.0.1.1:11211" },
      { "184769", "10.0.1.3:11211" }, { "185002", "10.0.1.1:11211" },
      { "185235", "10.0.1.6:11211" }, { "185468", "10.0.1.7:11211" },
      { "185701", "10.0.1.4:11211" }, { "185934", "10.0.1.1:11211" },
      { "186167", "10.0.1.8:11211" }, { "186400", "10.0.1.8:11211" },
      { "186633", "10.0.1.1:11211" }, { "186866", "10.0.1.5:11211" },
      { "187099", "10.0.1.8:11211" }, { "187332", "10.0.1.1:11211" },
      { "187565", "10.0.1.5:11211" }, { "187798", "10.0.1.1:11211" },
      { "188031", "10.0.1.8:11211" }, { "188264", "10.0.1.5:11211" },
      { "188497", "10.0.1.2:11211" }, { "188730", "10.0.1.6:11211" },
      { "196419", "10.0.1.8:11211" }, { "196652", "10.0.1.7:11211" },
      { "196885", "10.0.1.1:11211" }, { "197118", "10.0.1.6:11211" },
      { "197351", "10.0.1.1:11211" }, { "197584", "10.0.1.1:11211" },
      { "197817", "10.0.1.8:11211" }, { "198050", "10.0.1.6:11211" },
      { "198283", "10.0.1.3:11211" }, { "198516", "10.0.1.8:11211" },
      { "198749", "10.0.1.6:11211" }, { "198982", "10.0.1.2:11211" },
      { "199215", "10.0.1.4:11211" }, { "199448", "10.0.1.5:11211" },
      { "199681", "10.0.1.6:11211" }, { "199914", "10.0.1.6:11211" },
      { "200147", "10.0.1.3:11211" }, { "200380", "10.0.1.4:11211" },
      { "200613", "10.0.1.1:11211" }, { "200846", "10.0.1.6:11211" },
      { "201079", "10.0.1.7:11211" }, { "201312", "10.0.1.7:11211" },
      { "201545", "10.0.1.1:11211" }, { "201778", "10.0.1.1:11211" },
      { "202011", "10.0.1.7:11211" }, { "202244", "10.0.1.7:11211" },
      { "202477", "10.0.1.6:11211" }, { "202710", "10.0.1.1:11211" },
      { "202943", "10.0.1.1:11211" }, { "203176", "10.0.1.1:11211" },
      { "203409", "10.0.1.3:11211" }, { "203642", "10.0.1.5:11211" },
      { "203875", "10.0.1.1:11211" }, { "204108", "10.0.1.8:11211" },
      { "204341", "10.0.1.1:11211" }, { "204574", "10.0.1.4:11211" },
      { "204807", "10.0.1.3:11211" }, { "205040", "10.0.1.7:11211" },
      { "205273", "10.0.1.2:11211" }, { "205506", "10.0.1.6:11211" },
      { "205739", "10.0.1.2:11211" }, { "205972", "10.0.1.6:11211" },
      { "206205", "10.0.1.6:11211" }, { "206438", "10.0.1.6:11211" },
      { "206671", "10.0.1.7:11211" }, { "206904", "10.0.1.6:11211" },
      { "207137", "10.0.1.7:11211" }, { "207370", "10.0.1.5:11211" },
      { "207603", "10.0.1.1:11211" }, { "207836", "10.0.1.5:11211" },
      { "208069", "10.0.1.6:11211" }, { "208302", "10.0.1.8:11211" },
      { "208535", "10.0.1.5:11211" }, { "208768", "10.0.1.3:11211" },
      { "209001", "10.0.1.2:11211" }, { "209234", "10.0.1.3:11211" },
      { "209467", "10.0.1.4:11211" }, { "209700", "10.0.1.6:11211" },
      { "209933", "10.0.1.5:11211" }, { "210166", "10.0.1.1:11211" },
      { "210399", "10.0.1.8:11211" }, { "210632", "10.0.1.4:11211" },
      { "210865", "10.0.1.8:11211" }, { "211098", "10.0.1.8:11211" },
      { "211331", "10.0.1.3:11211" }, { "211564", "10.0.1.6:11211" },
      { "211797", "10.0.1.8:11211" }, { "212030", "10.0.1.8:11211" },
      { "212263", "10.0.1.7:11211" }, { "212496", "10.0.1.8:11211" },
      { "212729", "10.0.1.1:11211" }, { "212962", "10.0.1.7:11211" },
      { "213195", "10.0.1.6:11211" }, { "213428", "10.0.1.2:11211" },
      { "213661", "10.0.1.8:11211" }, { "213894", "10.0.1.5:11211" },
      { "214127", "10.0.1.3:11211" }, { "214360", "10.0.1.1:11211" },
      { "214593", "10.0.1.2:11211" }, { "214826", "10.0.1.7:11211" },
      { "215059", "10.0.1.7:11211" }, { "215292", "10.0.1.2:11211" },
      { "215525", "10.0.1.1:11211" }, { "215758", "10.0.1.8:11211" },
      { "215991", "10.0.1.3:11211" }, { "216224", "10.0.1.8:11211" },
      { "216457", "10.0.1.3:11211" }, { "216690", "10.0.1.7:11211" },
      { "216923", "10.0.1.2:11211" }, { "217156", "10.0.1.2:11211" },
      { "217389", "10.0.1.2:11211" }, { "217622", "10.0.1.4:11211" },
      { "217855", "10.0.1.3:11211" }, { "218088", "10.0.1.7:11211" },
      { "218321", "10.0.1.8:11211" }, { "218554", "10.0.1.7:11211" },
      { "218787", "10.0.1.2:11211" }, { "219020", "10.0.1.8:11211" },
      { "219253", "10.0.1.8:11211" }, { "219486", "10.0.1.5:11211" },
      { "219719", "10.0.1.2:11211" }, { "219952", "10.0.1.3:11211" },
      { "220185", "10.0.1.7:11211" }, { "220418", "10.0.1.7:11211" },
      { "220651", "10.0.1.7:11211" }, { "220884", "10.0.1.5:11211" },
      { "221117", "10.0.1.5:11211" }, { "221350", "10.0.1.8:11211" },
      { "221583", "10.0.1.7:11211" }, { "221816", "10.0.1.7:11211" },
      { "222049", "10.0.1.8:11211" }, { "222282", "10.0.1.2:11211" },
      { "222515", "10.0.1.8:11211" }, { "222748", "10.0.1.8:11211" },
      { "222981", "10.0.1.6:11211" }, { "223214", "10.0.1.2:11211" },
      { "223447", "10.0.1.4:11211" }, { "223680", "10.0.1.2:11211" },
      { "223913", "10.0.1.8:11211" }, { "224146", "10.0.1.8:11211" },
      { "224379", "10.0.1.3:11211" }, { "224612", "10.0.1.5:11211" },
      { "224845", "10.0.1.8:11211" }, { "225078", "10.0.1.8:11211" },
      { "225311", "10.0.1.3:11211" }, { "225544", "10.0.1.5:11211" },
      { "225777", "10.0.1.1:11211" }, { "226010", "10.0.1.6:11211" },
      { "226243", "10.0.1.7:11211" }, { "226476", "10.0.1.2:11211" },
      { "226709", "10.0.1.3:11211" }, { "226942", "10.0.1.5:11211" },
      { "227175", "10.0.1.3:11211" }, { "227408", "10.0.1.1:11211" },
      { "227641", "10.0.1.8:11211" }, { "227874", "10.0.1.4:11211" },
      { "228107", "10.0.1.6:11211" }, { "228340", "10.0.1.4:11211" },
      { "228573", "10.0.1.1:11211" }, { "228806", "10.0.1.3:11211" },
      { "238825", "10.0.1.2:11211" }, { "239058", "10.0.1.5:11211" },
      { "239291", "10.0.1.5:11211" }, { "239524", "10.0.1.6:11211" },
      { "239757", "10.0.1.5:11211" }, { "239990", "10.0.1.7:11211" },
      { "240223", "10.0.1.7:11211" }, { "240456", "10.0.1.5:11211" },
      { "240689", "10.0.1.5:11211" }, { "240922", "10.0.1.1:11211" },
      { "241155", "10.0.1.1:11211" }, { "241388", "10.0.1.7:11211" },
      { "241621", "10.0.1.1:11211" }, { "241854", "10.0.1.8:11211" },
      { "242087", "10.0.1.6:11211" }, { "242320", "10.0.1.8:11211" },
      { "242553", "10.0.1.2:11211" }, { "242786", "10.0.1.5:11211" },
      { "243019", "10.0.1.6:11211" }, { "243252", "10.0.1.4:11211" },
      { "243485", "10.0.1.8:11211" }, { "243718", "10.0.1.7:11211" },
      { "243951", "10.0.1.5:11211" }, { "249310", "10.0.1.5:11211" },
      { "249543", "10.0.1.8:11211" }, { "249776", "10.0.1.7:11211" },
      { "250009", "10.0.1.3:11211" }, { "250242", "10.0.1.8:11211" },
      { "250475", "10.0.1.4:11211" }, { "250708", "10.0.1.8:11211" },
      { "250941", "10.0.1.1:11211" }, { "251174", "10.0.1.5:11211" },
      { "251407", "10.0.1.8:11211" }, { "251640", "10.0.1.1:11211" },
      { "251873", "10.0.1.2:11211" }, { "252106", "10.0.1.5:11211" },
      { "252339", "10.0.1.2:11211" }, { "252572", "10.0.1.5:11211" },
      { "252805", "10.0.1.8:11211" }, { "253038", "10.0.1.5:11211" },
      { "253271", "10.0.1.7:11211" }, { "253504", "10.0.1.4:11211" },
      { "253737", "10.0.1.3:11211" }, { "253970", "10.0.1.7:11211" },
      { "254203", "10.0.1.6:11211" }, { "254436", "10.0.1.2:11211" },
      { "254669", "10.0.1.5:11211" }, { "254902", "10.0.1.8:11211" },
      { "255135", "10.0.1.5:11211" }, { "255368", "10.0.1.3:11211" },
      { "255601", "10.0.1.4:11211" }, { "255834", "10.0.1.1:11211" },
      { "256067", "10.0.1.8:11211" }, { "260960", "10.0.1.5:11211" },
      { "261193", "10.0.1.5:11211" }, { "261426", "10.0.1.2:11211" },
      { "261659", "10.0.1.2:11211" }, { "261892", "10.0.1.2:11211" },
      { "262125", "10.0.1.8:11211" }, { "262358", "10.0.1.6:11211" },
      { "262591", "10.0.1.8:11211" }, { "262824", "10.0.1.1:11211" },
      { "263057", "10.0.1.5:11211" }, { "263290", "10.0.1.1:11211" },
      { "263523", "10.0.1.3:11211" }, { "263756", "10.0.1.2:11211" },
      { "263989", "10.0.1.5:11211" }, { "264222", "10.0.1.5:11211" },
      { "264455", "10.0.1.5:11211" }, { "264688", "10.0.1.5:11211" },
      { "264921", "10.0.1.6:11211" }, { "265154", "10.0.1.7:11211" },
      { "265387", "10.0.1.7:11211" }, { "265620", "10.0.1.4:11211" },
      { "265853", "10.0.1.5:11211" }, { "266086", "10.0.1.6:11211" },
      { "266319", "10.0.1.1:11211" }, { "266552", "10.0.1.2:11211" },
      { "266785", "10.0.1.7:11211" }, { "267018", "10.0.1.1:11211" },
      { "267251", "10.0.1.6:11211" }, { "267484", "10.0.1.8:11211" },
      { "267717", "10.0.1.3:11211" }, { "267950", "10.0.1.2:11211" },
      { "268183", "10.0.1.5:11211" }, { "268416", "10.0.1.6:11211" },
      { "268649", "10.0.1.5:11211" }, { "268882", "10.0.1.6:11211" },
      { "269115", "10.0.1.5:11211" }, { "269348", "10.0.1.2:11211" },
      { "269581", "10.0.1.4:11211" }, { "269814", "10.0.1.6:11211" },
      { "270047", "10.0.1.2:11211" }, { "270280", "10.0.1.1:11211" },
      { "270513", "10.0.1.8:11211" }, { "270746", "10.0.1.6:11211" },
      { "270979", "10.0.1.4:11211" }, { "271212", "10.0.1.6:11211" },
      { "271445", "10.0.1.8:11211" }, { "271678", "10.0.1.7:11211" },
      { "271911", "10.0.1.3:11211" }, { "272144", "10.0.1.8:11211" },
      { "272377", "10.0.1.7:11211" }, { "272610", "10.0.1.1:11211" },
      { "272843", "10.0.1.8:11211" }, { "273076", "10.0.1.8:11211" },
      { "273309", "10.0.1.1:11211" }, { "273542", "10.0.1.8:11211" },
      { "273775", "10.0.1.5:11211" }, { "274008", "10.0.1.7:11211" },
      { "274241", "10.0.1.7:11211" }, { "274474", "10.0.1.6:11211" },
      { "274707", "10.0.1.6:11211" }, { "274940", "10.0.1.8:11211" },
      { "275173", "10.0.1.4:11211" }, { "275406", "10.0.1.6:11211" },
      { "275639", "10.0.1.2:11211" }, { "275872", "10.0.1.3:11211" },
      { "276105", "10.0.1.5:11211" }, { "276338", "10.0.1.2:11211" },
      { "276571", "10.0.1.4:11211" }, { "276804", "10.0.1.4:11211" },
      { "277037", "10.0.1.6:11211" }, { "277270", "10.0.1.6:11211" },
      { "277503", "10.0.1.1:11211" }, { "277736", "10.0.1.6:11211" },
      { "277969", "10.0.1.7:11211" }, { "278202", "10.0.1.3:11211" },
      { "278435", "10.0.1.6:11211" }, { "278668", "10.0.1.3:11211" },
      { "278901", "10.0.1.1:11211" }, { "279134", "10.0.1.7:11211" },
      { "279367", "10.0.1.4:11211" }, { "279600", "10.0.1.6:11211" },
      { "279833", "10.0.1.8:11211" }, { "280066", "10.0.1.7:11211" },
      { "280299", "10.0.1.8:11211" }, { "280532", "10.0.1.5:11211" },
      { "280765", "10.0.1.8:11211" }, { "280998", "10.0.1.1:11211" },
      { "281231", "10.0.1.5:11211" }, { "281464", "10.0.1.8:11211" },
      { "281697", "10.0.1.2:11211" }, { "281930", "10.0.1.7:11211" },
      { "282163", "10.0.1.5:11211" }, { "282396", "10.0.1.4:11211" },
      { "282629", "10.0.1.7:11211" }, { "282862", "10.0.1.7:11211" },
      { "283095", "10.0.1.5:11211" }, { "283328", "10.0.1.2:11211" },
      { "283561", "10.0.1.1:11211" }, { "283794", "10.0.1.8:11211" },
      { "284027", "10.0.1.2:11211" }, { "284260", "10.0.1.6:11211" },
      { "284493", "10.0.1.2:11211" }, { "284726", "10.0.1.2:11211" },
      { "284959", "10.0.1.6:11211" }, { "285192", "10.0.1.3:11211" },
      { "285425", "10.0.1.6:11211" }, { "285658", "10.0.1.2:11211" },
      { "285891", "10.0.1.8:11211" }, { "286124", "10.0.1.3:11211" },
      { "286357", "10.0.1.4:11211" }, { "286590", "10.0.1.5:11211" },
      { "286823", "10.0.1.7:11211" }, { "287056", "10.0.1.6:11211" },
      { "287289", "10.0.1.3:11211" }, { "287522", "10.0.1.3:11211" },
      { "287755", "10.0.1.3:11211" }, { "287988", "10.0.1.6:11211" },
      { "288221", "10.0.1.8:11211" }, { "288454", "10.0.1.4:11211" },
      { "288687", "10.0.1.3:11211" }, { "288920", "10.0.1.5:11211" },
      { "289153", "10.0.1.8:11211" }, { "289386", "10.0.1.7:11211" },
      { "289619", "10.0.1.8:11211" }, { "289852", "10.0.1.8:11211" },
      { "290085", "10.0.1.3:11211" }, { "290318", "10.0.1.7:11211" },
      { "290551", "10.0.1.8:11211" }, { "290784", "10.0.1.7:11211" },
      { "291017", "10.0.1.5:11211" }, { "291250", "10.0.1.8:11211" },
      { "291483", "10.0.1.3:11211" }, { "291716", "10.0.1.3:11211" },
      { "291949", "10.0.1.1:11211" }, { "292182", "10.0.1.8:11211" },
      { "292415", "10.0.1.5:11211" }, { "292648", "10.0.1.8:11211" },
      { "292881", "10.0.1.5:11211" }, { "293114", "10.0.1.7:11211" },
      { "293347", "10.0.1.5:11211" }, { "293580", "10.0.1.5:11211" },
      { "293813", "10.0.1.5:11211" }, { "294046", "10.0.1.3:11211" },
      { "294279", "10.0.1.6:11211" }, { "294512", "10.0.1.8:11211" },
      { "294745", "10.0.1.7:11211" }, { "294978", "10.0.1.7:11211" },
      { "295211", "10.0.1.2:11211" }, { "295444", "10.0.1.5:11211" },
      { "295677", "10.0.1.7:11211" }, { "295910", "10.0.1.3:11211" },
      { "296143", "10.0.1.7:11211" }, { "296376", "10.0.1.4:11211" },
      { "296609", "10.0.1.3:11211" }, { "296842", "10.0.1.6:11211" },
      { "297075", "10.0.1.2:11211" }, { "297308", "10.0.1.4:11211" },
      { "297541", "10.0.1.8:11211" }, { "297774", "10.0.1.3:11211" },
      { "298007", "10.0.1.3:11211" }, { "298240", "10.0.1.5:11211" },
      { "298473", "10.0.1.4:11211" }, { "298706", "10.0.1.5:11211" },
      { "298939", "10.0.1.7:11211" }, { "303599", "10.0.1.6:11211" },
      { "303832", "10.0.1.4:11211" }, { "304065", "10.0.1.3:11211" },
      { "304298", "10.0.1.1:11211" }, { "304531", "10.0.1.1:11211" },
      { "304764", "10.0.1.8:11211" }, { "304997", "10.0.1.7:11211" },
      { "305230", "10.0.1.6:11211" }, { "305463", "10.0.1.2:11211" },
      { "305696", "10.0.1.7:11211" }, { "305929", "10.0.1.4:11211" },
      { "306162", "10.0.1.4:11211" }, { "306395", "10.0.1.4:11211" },
      { "306628", "10.0.1.8:11211" }, { "306861", "10.0.1.7:11211" },
      { "307094", "10.0.1.4:11211" }, { "307327", "10.0.1.8:11211" },
      { "307560", "10.0.1.2:11211" }, { "307793", "10.0.1.8:11211" },
      { "308026", "10.0.1.5:11211" }, { "308259", "10.0.1.6:11211" },
      { "308492", "10.0.1.2:11211" }, { "308725", "10.0.1.6:11211" },
      { "308958", "10.0.1.3:11211" }, { "309191", "10.0.1.7:11211" },
      { "309424", "10.0.1.7:11211" }, { "309657", "10.0.1.8:11211" },
      { "309890", "10.0.1.8:11211" }, { "310123", "10.0.1.1:11211" },
      { "310356", "10.0.1.3:11211" }, { "310589", "10.0.1.5:11211" },
      { "310822", "10.0.1.8:11211" }, { "311055", "10.0.1.3:11211" },
      { "311288", "10.0.1.8:11211" }, { "311521", "10.0.1.8:11211" },
      { "311754", "10.0.1.3:11211" }, { "311987", "10.0.1.7:11211" },
      { "312220", "10.0.1.5:11211" }, { "312453", "10.0.1.2:11211" },
      { "312686", "10.0.1.2:11211" }, { "312919", "10.0.1.1:11211" },
      { "313152", "10.0.1.8:11211" }, { "313385", "10.0.1.2:11211" },
      { "313618", "10.0.1.8:11211" }, { "313851", "10.0.1.4:11211" },
      { "314084", "10.0.1.4:11211" }, { "314317", "10.0.1.6:11211" },
      { "314550", "10.0.1.5:11211" }, { "314783", "10.0.1.1:11211" },
      { "315016", "10.0.1.1:11211" }, { "315249", "10.0.1.6:11211" },
      { "315482", "10.0.1.4:11211" }, { "315715", "10.0.1.2:11211" },
      { "315948", "10.0.1.7:11211" }, { "316181", "10.0.1.2:11211" },
      { "316414", "10.0.1.8:11211" }, { "316647", "10.0.1.7:11211" },
      { "316880", "10.0.1.7:11211" }, { "317113", "10.0.1.8:11211" },
      { "317346", "10.0.1.7:11211" }, { "317579", "10.0.1.4:11211" },
      { "317812", "10.0.1.2:11211" }, { "318045", "10.0.1.1:11211" },
      { "318278", "10.0.1.6:11211" }, { "318511", "10.0.1.6:11211" },
      { "318744", "10.0.1.1:11211" }, { "318977", "10.0.1.5:11211" },
      { "319210", "10.0.1.2:11211" }, { "319443", "10.0.1.4:11211" },
      { "319676", "10.0.1.7:11211" }, { "319909", "10.0.1.3:11211" },
      { "320142", "10.0.1.8:11211" }, { "320375", "10.0.1.1:11211" },
      { "320608", "10.0.1.8:11211" }, { "320841", "10.0.1.1:11211" },
      { "321074", "10.0.1.5:11211" }, { "321307", "10.0.1.2:11211" },
      { "321540", "10.0.1.3:11211" }, { "321773", "10.0.1.6:11211" },
      { "322006", "10.0.1.3:11211" }, { "322239", "10.0.1.7:11211" },
      { "322472", "10.0.1.6:11211" }, { "322705", "10.0.1.8:11211" },
      { "322938", "10.0.1.4:11211" }, { "323171", "10.0.1.7:11211" },
      { "323404", "10.0.1.2:11211" }, { "323637", "10.0.1.4:11211" },
      { "323870", "10.0.1.2:11211" }, { "324103", "10.0.1.7:11211" },
      { "324336", "10.0.1.3:11211" }, { "324569", "10.0.1.2:11211" },
      { "324802", "10.0.1.3:11211" }, { "325035", "10.0.1.8:11211" },
      { "325268", "10.0.1.5:11211" }, { "325501", "10.0.1.6:11211" },
      { "325734", "10.0.1.3:11211" }, { "325967", "10.0.1.5:11211" },
      { "326200", "10.0.1.1:11211" }, { "326433", "10.0.1.8:11211" },
      { "326666", "10.0.1.4:11211" }, { "326899", "10.0.1.2:11211" },
      { "327132", "10.0.1.4:11211" }, { "327365", "10.0.1.1:11211" },
      { "327598", "10.0.1.3:11211" }, { "327831", "10.0.1.6:11211" },
      { "328064", "10.0.1.4:11211" }, { "328297", "10.0.1.6:11211" },
      { "328530", "10.0.1.8:11211" }, { "328763", "10.0.1.2:11211" },
      { "328996", "10.0.1.1:11211" }, { "329229", "10.0.1.3:11211" },
      { "329462", "10.0.1.7:11211" }, { "329695", "10.0.1.6:11211" },
      { "329928", "10.0.1.1:11211" }, { "330161", "10.0.1.7:11211" },
      { "330394", "10.0.1.1:11211" }, { "330627", "10.0.1.3:11211" },
      { "330860", "10.0.1.8:11211" }, { "331093", "10.0.1.2:11211" },
      { "331326", "10.0.1.6:11211" }, { "331559", "10.0.1.4:11211" },
      { "331792", "10.0.1.7:11211" }, { "332025", "10.0.1.8:11211" },
      { "332258", "10.0.1.2:11211" }, { "332491", "10.0.1.5:11211" },
      { "332724", "10.0.1.6:11211" }, { "332957", "10.0.1.4:11211" },
      { "333190", "10.0.1.8:11211" }, { "333423", "10.0.1.7:11211" },
      { "333656", "10.0.1.7:11211" }, { "333889", "10.0.1.1:11211" },
      { "334122", "10.0.1.5:11211" }, { "334355", "10.0.1.4:11211" },
      { "334588", "10.0.1.8:11211" }, { "334821", "10.0.1.3:11211" },
      { "335054", "10.0.1.3:11211" }, { "335287", "10.0.1.7:11211" },
      { "335520", "10.0.1.8:11211" }, { "335753", "10.0.1.2:11211" },
      { "335986", "10.0.1.3:11211" }, { "336219", "10.0.1.2:11211" },
      { "336452", "10.0.1.8:11211" }, { "336685", "10.0.1.2:11211" },
      { "336918", "10.0.1.7:11211" }, { "337151", "10.0.1.4:11211" },
      { "337384", "10.0.1.6:11211" }, { "337617", "10.0.1.1:11211" },
      { "337850", "10.0.1.2:11211" }, { "338083", "10.0.1.7:11211" },
      { "338316", "10.0.1.8:11211" }, { "338549", "10.0.1.5:11211" },
      { "338782", "10.0.1.8:11211" }, { "339015", "10.0.1.8:11211" },
      { "339248", "10.0.1.5:11211" }, { "339481", "10.0.1.8:11211" },
      { "339714", "10.0.1.3:11211" }, { "339947", "10.0.1.7:11211" },
      { "340180", "10.0.1.5:11211" }, { "340413", "10.0.1.5:11211" },
      { "340646", "10.0.1.8:11211" }, { "340879", "10.0.1.3:11211" },
      { "341112", "10.0.1.6:11211" }, { "341345", "10.0.1.8:11211" },
      { "341578", "10.0.1.7:11211" }, { "341811", "10.0.1.2:11211" },
      { "342044", "10.0.1.1:11211" }, { "342277", "10.0.1.4:11211" },
      { "342510", "10.0.1.5:11211" }, { "342743", "10.0.1.5:11211" },
      { "342976", "10.0.1.4:11211" }, { "343209", "10.0.1.7:11211" },
      { "343442", "10.0.1.8:11211" }, { "343675", "10.0.1.1:11211" },
      { "343908", "10.0.1.2:11211" }, { "348801", "10.0.1.2:11211" },
      { "349034", "10.0.1.7:11211" }, { "349267", "10.0.1.5:11211" },
      { "349500", "10.0.1.7:11211" }, { "349733", "10.0.1.8:11211" },
      { "349966", "10.0.1.3:11211" }, { "350199", "10.0.1.8:11211" },
      { "350432", "10.0.1.4:11211" }, { "350665", "10.0.1.5:11211" },
      { "350898", "10.0.1.2:11211" }, { "351131", "10.0.1.5:11211" },
      { "351364", "10.0.1.8:11211" }, { "351597", "10.0.1.3:11211" },
      { "351830", "10.0.1.5:11211" }, { "352063", "10.0.1.3:11211" },
      { "352296", "10.0.1.1:11211" }, { "352529", "10.0.1.4:11211" },
      { "352762", "10.0.1.5:11211" }, { "352995", "10.0.1.1:11211" },
      { "353228", "10.0.1.8:11211" }, { "357888", "10.0.1.5:11211" },
      { "358121", "10.0.1.8:11211" }, { "358354", "10.0.1.1:11211" },
      { "358587", "10.0.1.1:11211" }, { "358820", "10.0.1.7:11211" },
      { "359053", "10.0.1.4:11211" }, { "359286", "10.0.1.2:11211" },
      { "359519", "10.0.1.3:11211" }, { "359752", "10.0.1.1:11211" },
      { "359985", "10.0.1.1:11211" }, { "360218", "10.0.1.4:11211" },
      { "360451", "10.0.1.5:11211" }, { "360684", "10.0.1.2:11211" },
      { "360917", "10.0.1.4:11211" }, { "361150", "10.0.1.1:11211" },
      { "361383", "10.0.1.2:11211" }, { "361616", "10.0.1.3:11211" },
      { "361849", "10.0.1.3:11211" }, { "362082", "10.0.1.2:11211" },
      { "362315", "10.0.1.4:11211" }, { "362548", "10.0.1.5:11211" },
      { "362781", "10.0.1.6:11211" }, { "363014", "10.0.1.7:11211" },
      { "363247", "10.0.1.8:11211" }, { "363480", "10.0.1.7:11211" },
      { "363713", "10.0.1.4:11211" }, { "363946", "10.0.1.4:11211" },
      { "364179", "10.0.1.2:11211" }, { "364412", "10.0.1.5:11211" },
      { "364645", "10.0.1.8:11211" }, { "364878", "10.0.1.1:11211" },
      { "365111", "10.0.1.6:11211" }, { "365344", "10.0.1.5:11211" },
      { "365577", "10.0.1.6:11211" }, { "365810", "10.0.1.8:11211" },
      { "366043", "10.0.1.5:11211" }, { "366276", "10.0.1.4:11211" },
      { "366509", "10.0.1.6:11211" }, { "366742", "10.0.1.7:11211" },
      { "374431", "10.0.1.4:11211" }, { "374664", "10.0.1.8:11211" },
      { "374897", "10.0.1.6:11211" }, { "375130", "10.0.1.3:11211" },
      { "375363", "10.0.1.4:11211" }, { "375596", "10.0.1.8:11211" },
      { "375829", "10.0.1.1:11211" }, { "376062", "10.0.1.3:11211" },
      { "376295", "10.0.1.1:11211" }, { "376528", "10.0.1.1:11211" },
      { "376761", "10.0.1.4:11211" }, { "376994", "10.0.1.7:11211" },
      { "377227", "10.0.1.2:11211" }, { "377460", "10.0.1.8:11211" },
      { "377693", "10.0.1.1:11211" }, { "377926", "10.0.1.5:11211" },
      { "378159", "10.0.1.1:11211" }, { "378392", "10.0.1.8:11211" },
      { "378625", "10.0.1.7:11211" }, { "378858", "10.0.1.4:11211" },
      { "379091", "10.0.1.3:11211" }, { "379324", "10.0.1.8:11211" },
      { "379557", "10.0.1.2:11211" }, { "379790", "10.0.1.2:11211" },
      { "380023", "10.0.1.8:11211" }, { "380256", "10.0.1.6:11211" },
      { "380489", "10.0.1.2:11211" }, { "380722", "10.0.1.8:11211" },
      { "380955", "10.0.1.5:11211" }, { "381188", "10.0.1.8:11211" },
      { "381421", "10.0.1.8:11211" }, { "381654", "10.0.1.4:11211" },
      { "381887", "10.0.1.7:11211" }, { "382120", "10.0.1.7:11211" },
      { "382353", "10.0.1.3:11211" }, { "382586", "10.0.1.6:11211" },
      { "382819", "10.0.1.2:11211" }, { "383052", "10.0.1.8:11211" },
      { "383285", "10.0.1.7:11211" }, { "383518", "10.0.1.8:11211" },
      { "383751", "10.0.1.2:11211" }, { "383984", "10.0.1.6:11211" },
      { "384217", "10.0.1.6:11211" }, { "384450", "10.0.1.3:11211" },
      { "384683", "10.0.1.8:11211" }, { "384916", "10.0.1.3:11211" },
      { "385149", "10.0.1.1:11211" }, { "385382", "10.0.1.2:11211" },
      { "385615", "10.0.1.1:11211" }, { "385848", "10.0.1.5:11211" },
      { "386081", "10.0.1.5:11211" }, { "386314", "10.0.1.6:11211" },
      { "386547", "10.0.1.3:11211" }, { "386780", "10.0.1.4:11211" },
      { "387013", "10.0.1.7:11211" }, { "387246", "10.0.1.8:11211" },
      { "387479", "10.0.1.3:11211" }, { "387712", "10.0.1.8:11211" },
      { "387945", "10.0.1.8:11211" }, { "388178", "10.0.1.6:11211" },
      { "388411", "10.0.1.1:11211" }, { "388644", "10.0.1.8:11211" },
      { "388877", "10.0.1.7:11211" }, { "389110", "10.0.1.1:11211" },
      { "389343", "10.0.1.1:11211" }, { "389576", "10.0.1.8:11211" },
      { "389809", "10.0.1.2:11211" }, { "390042", "10.0.1.5:11211" },
      { "390275", "10.0.1.7:11211" }, { "390508", "10.0.1.2:11211" },
      { "390741", "10.0.1.2:11211" }, { "390974", "10.0.1.4:11211" },
      { "391207", "10.0.1.3:11211" }, { "391440", "10.0.1.4:11211" },
      { "391673", "10.0.1.4:11211" }, { "391906", "10.0.1.7:11211" },
      { "392139", "10.0.1.3:11211" }, { "392372", "10.0.1.6:11211" },
      { "392605", "10.0.1.6:11211" }, { "392838", "10.0.1.6:11211" },
      { "393071", "10.0.1.2:11211" }, { "393304", "10.0.1.2:11211" },
      { "393537", "10.0.1.2:11211" }, { "393770", "10.0.1.4:11211" },
      { "394003", "10.0.1.1:11211" }, { "394236", "10.0.1.6:11211" },
      { "394469", "10.0.1.4:11211" }, { "394702", "10.0.1.6:11211" },
      { "394935", "10.0.1.7:11211" }, { "395168", "10.0.1.4:11211" },
      { "395401", "10.0.1.3:11211" }, { "395634", "10.0.1.4:11211" },
      { "395867", "10.0.1.1:11211" }, { "396100", "10.0.1.6:11211" },
      { "396333", "10.0.1.4:11211" }, { "396566", "10.0.1.8:11211" },
      { "396799", "10.0.1.3:11211" }, { "397032", "10.0.1.4:11211" },
      { "397265", "10.0.1.3:11211" }, { "397498", "10.0.1.5:11211" },
      { "397731", "10.0.1.1:11211" }, { "397964", "10.0.1.2:11211" },
      { "398197", "10.0.1.8:11211" }, { "398430", "10.0.1.7:11211" },
      { "398663", "10.0.1.1:11211" }, { "398896", "10.0.1.5:11211" },
      { "399129", "10.0.1.5:11211" }, { "399362", "10.0.1.8:11211" },
      { "399595", "10.0.1.8:11211" }, { "399828", "10.0.1.7:11211" },
      { "400061", "10.0.1.6:11211" }, { "400294", "10.0.1.8:11211" },
      { "400527", "10.0.1.2:11211" }, { "400760", "10.0.1.7:11211" },
      { "400993", "10.0.1.7:11211" }, { "401226", "10.0.1.1:11211" },
      { "401459", "10.0.1.1:11211" }, { "401692", "10.0.1.2:11211" },
      { "401925", "10.0.1.1:11211" }, { "402158", "10.0.1.1:11211" },
      { "402391", "10.0.1.5:11211" }, { "402624", "10.0.1.1:11211" },
      { "402857", "10.0.1.8:11211" }, { "403090", "10.0.1.1:11211" },
      { "403323", "10.0.1.5:11211" }, { "403556", "10.0.1.6:11211" },
      { "403789", "10.0.1.6:11211" }, { "404022", "10.0.1.2:11211" },
      { "404255", "10.0.1.4:11211" }, { "404488", "10.0.1.1:11211" },
      { "404721", "10.0.1.8:11211" }, { "404954", "10.0.1.1:11211" },
      { "405187", "10.0.1.7:11211" }, { "405420", "10.0.1.7:11211" },
      { "405653", "10.0.1.7:11211" }, { "405886", "10.0.1.1:11211" },
      { "406119", "10.0.1.3:11211" }, { "406352", "10.0.1.5:11211" },
      { "411944", "10.0.1.1:11211" }, { "412177", "10.0.1.8:11211" },
      { "412410", "10.0.1.3:11211" }, { "412643", "10.0.1.7:11211" },
      { "412876", "10.0.1.4:11211" }, { "413109", "10.0.1.8:11211" },
      { "413342", "10.0.1.7:11211" }, { "413575", "10.0.1.8:11211" },
      { "413808", "10.0.1.7:11211" }, { "414041", "10.0.1.1:11211" },
      { "414274", "10.0.1.5:11211" }, { "414507", "10.0.1.1:11211" },
      { "414740", "10.0.1.8:11211" }, { "414973", "10.0.1.2:11211" },
      { "415206", "10.0.1.8:11211" }, { "415439", "10.0.1.6:11211" },
      { "415672", "10.0.1.1:11211" }, { "415905", "10.0.1.7:11211" },
      { "416138", "10.0.1.6:11211" }, { "416371", "10.0.1.8:11211" },
      { "416604", "10.0.1.4:11211" }, { "416837", "10.0.1.4:11211" },
      { "417070", "10.0.1.4:11211" }, { "417303", "10.0.1.6:11211" },
      { "417536", "10.0.1.6:11211" }, { "417769", "10.0.1.8:11211" },
      { "418002", "10.0.1.3:11211" }, { "418235", "10.0.1.8:11211" },
      { "418468", "10.0.1.4:11211" }, { "418701", "10.0.1.4:11211" },
      { "418934", "10.0.1.7:11211" }, { "419167", "10.0.1.6:11211" },
      { "419400", "10.0.1.3:11211" }, { "419633", "10.0.1.5:11211" },
      { "419866", "10.0.1.8:11211" }, { "420099", "10.0.1.6:11211" },
      { "420332", "10.0.1.3:11211" }, { "420565", "10.0.1.5:11211" },
      { "420798", "10.0.1.8:11211" }, { "421031", "10.0.1.8:11211" },
      { "421264", "10.0.1.6:11211" }, { "426856", "10.0.1.3:11211" },
      { "427089", "10.0.1.5:11211" }, { "427322", "10.0.1.1:11211" },
      { "427555", "10.0.1.6:11211" }, { "427788", "10.0.1.5:11211" },
      { "428021", "10.0.1.8:11211" }, { "428254", "10.0.1.3:11211" },
      { "428487", "10.0.1.7:11211" }, { "428720", "10.0.1.4:11211" },
      { "428953", "10.0.1.4:11211" }, { "429186", "10.0.1.8:11211" },
      { "429419", "10.0.1.6:11211" }, { "429652", "10.0.1.8:11211" },
      { "429885", "10.0.1.6:11211" }, { "430118", "10.0.1.5:11211" },
      { "430351", "10.0.1.4:11211" }, { "430584", "10.0.1.6:11211" },
      { "430817", "10.0.1.5:11211" }, { "431050", "10.0.1.3:11211" },
      { "431283", "10.0.1.8:11211" }, { "431516", "10.0.1.1:11211" },
      { "431749", "10.0.1.4:11211" }, { "431982", "10.0.1.5:11211" },
      { "432215", "10.0.1.6:11211" }, { "432448", "10.0.1.6:11211" },
      { "432681", "10.0.1.4:11211" }, { "432914", "10.0.1.2:11211" },
      { "433147", "10.0.1.7:11211" }, { "433380", "10.0.1.6:11211" },
      { "433613", "10.0.1.1:11211" }, { "433846", "10.0.1.3:11211" },
      { "434079", "10.0.1.1:11211" }, { "434312", "10.0.1.2:11211" },
      { "434545", "10.0.1.4:11211" }, { "434778", "10.0.1.3:11211" },
      { "435011", "10.0.1.5:11211" }, { "435244", "10.0.1.2:11211" },
      { "435477", "10.0.1.4:11211" }, { "435710", "10.0.1.5:11211" },
      { "435943", "10.0.1.6:11211" }, { "436176", "10.0.1.8:11211" },
      { "436409", "10.0.1.5:11211" }, { "436642", "10.0.1.1:11211" },
      { "436875", "10.0.1.8:11211" }, { "437108", "10.0.1.6:11211" },
      { "437341", "10.0.1.2:11211" }, { "437574", "10.0.1.1:11211" },
      { "437807", "10.0.1.8:11211" }, { "438040", "10.0.1.7:11211" },
      { "438273", "10.0.1.3:11211" }, { "438506", "10.0.1.4:11211" },
      { "438739", "10.0.1.8:11211" }, { "438972", "10.0.1.7:11211" },
      { "439205", "10.0.1.8:11211" }, { "439438", "10.0.1.1:11211" },
      { "439671", "10.0.1.3:11211" }, { "439904", "10.0.1.3:11211" },
      { "440137", "10.0.1.4:11211" }, { "440370", "10.0.1.3:11211" },
      { "457612", "10.0.1.8:11211" }, { "457845", "10.0.1.3:11211" },
      { "458078", "10.0.1.1:11211" }, { "458311", "10.0.1.3:11211" },
      { "458544", "10.0.1.7:11211" }, { "458777", "10.0.1.7:11211" },
      { "459010", "10.0.1.5:11211" }, { "459243", "10.0.1.6:11211" },
      { "459476", "10.0.1.6:11211" }, { "459709", "10.0.1.6:11211" },
      { "459942", "10.0.1.1:11211" }, { "460175", "10.0.1.1:11211" },
      { "460408", "10.0.1.7:11211" }, { "460641", "10.0.1.5:11211" },
      { "460874", "10.0.1.5:11211" }, { "461107", "10.0.1.3:11211" },
      { "461340", "10.0.1.4:11211" }, { "461573", "10.0.1.7:11211" },
      { "461806", "10.0.1.7:11211" }, { "462039", "10.0.1.4:11211" },
      { "462272", "10.0.1.1:11211" }, { "462505", "10.0.1.8:11211" },
      { "462738", "10.0.1.5:11211" }, { "462971", "10.0.1.3:11211" },
      { "463204", "10.0.1.8:11211" }, { "463437", "10.0.1.8:11211" },
      { "463670", "10.0.1.5:11211" }, { "463903", "10.0.1.1:11211" },
      { "464136", "10.0.1.5:11211" }, { "464369", "10.0.1.6:11211" },
      { "464602", "10.0.1.7:11211" }, { "464835", "10.0.1.2:11211" },
      { "465068", "10.0.1.2:11211" }, { "465301", "10.0.1.1:11211" },
      { "465534", "10.0.1.8:11211" }, { "465767", "10.0.1.6:11211" },
      { "466000", "10.0.1.1:11211" }, { "466233", "10.0.1.6:11211" },
      { "466466", "10.0.1.3:11211" }, { "466699", "10.0.1.6:11211" },
      { "466932", "10.0.1.4:11211" }, { "467165", "10.0.1.4:11211" },
      { "467398", "10.0.1.2:11211" }, { "467631", "10.0.1.7:11211" },
      { "467864", "10.0.1.3:11211" }, { "468097", "10.0.1.6:11211" },
      { "468330", "10.0.1.1:11211" }, { "468563", "10.0.1.1:11211" },
      { "468796", "10.0.1.7:11211" }, { "469029", "10.0.1.6:11211" },
      { "469262", "10.0.1.3:11211" }, { "469495", "10.0.1.6:11211" },
      { "469728", "10.0.1.6:11211" }, { "469961", "10.0.1.4:11211" },
      { "470194", "10.0.1.5:11211" }, { "470427", "10.0.1.6:11211" },
      { "470660", "10.0.1.1:11211" }, { "470893", "10.0.1.4:11211" },
      { "471126", "10.0.1.7:11211" }, { "471359", "10.0.1.7:11211" },
      { "471592", "10.0.1.1:11211" }, { "471825", "10.0.1.5:11211" },
      { "472058", "10.0.1.6:11211" }, { "472291", "10.0.1.7:11211" },
      { "472524", "10.0.1.8:11211" }, { "472757", "10.0.1.2:11211" },
      { "472990", "10.0.1.2:11211" }, { "473223", "10.0.1.7:11211" },
      { "473456", "10.0.1.7:11211" }, { "473689", "10.0.1.5:11211" },
      { "473922", "10.0.1.7:11211" }, { "474155", "10.0.1.6:11211" },
      { "474388", "10.0.1.8:11211" }, { "474621", "10.0.1.3:11211" },
      { "474854", "10.0.1.1:11211" }, { "475087", "10.0.1.1:11211" },
      { "475320", "10.0.1.4:11211" }, { "475553", "10.0.1.6:11211" },
      { "475786", "10.0.1.4:11211" }, { "476019", "10.0.1.5:11211" },
      { "476252", "10.0.1.1:11211" }, { "476485", "10.0.1.4:11211" },
      { "476718", "10.0.1.6:11211" }, { "476951", "10.0.1.8:11211" },
      { "477184", "10.0.1.8:11211" }, { "477417", "10.0.1.3:11211" },
      { "477650", "10.0.1.1:11211" }, { "477883", "10.0.1.1:11211" },
      { "478116", "10.0.1.2:11211" }, { "478349", "10.0.1.5:11211" },
      { "478582", "10.0.1.4:11211" }, { "478815", "10.0.1.4:11211" },
      { "479048", "10.0.1.1:11211" }, { "479281", "10.0.1.1:11211" },
      { "479514", "10.0.1.2:11211" }, { "479747", "10.0.1.5:11211" },
      { "479980", "10.0.1.7:11211" }, { "480213", "10.0.1.3:11211" },
      { "480446", "10.0.1.1:11211" }, { "480679", "10.0.1.4:11211" },
      { "480912", "10.0.1.6:11211" }, { "481145", "10.0.1.2:11211" },
      { "481378", "10.0.1.1:11211" }, { "481611", "10.0.1.7:11211" },
      { "481844", "10.0.1.3:11211" }, { "482077", "10.0.1.4:11211" },
      { "482310", "10.0.1.4:11211" }, { "482543", "10.0.1.3:11211" },
      { "482776", "10.0.1.8:11211" }, { "483009", "10.0.1.7:11211" },
      { "483242", "10.0.1.6:11211" }, { "483475", "10.0.1.3:11211" },
      { "483708", "10.0.1.5:11211" }, { "483941", "10.0.1.8:11211" },
      { "484174", "10.0.1.7:11211" }, { "484407", "10.0.1.2:11211" },
      { "484640", "10.0.1.8:11211" }, { "484873", "10.0.1.8:11211" },
      { "485106", "10.0.1.4:11211" }, { "485339", "10.0.1.2:11211" },
      { "485572", "10.0.1.6:11211" }, { "485805", "10.0.1.5:11211" },
      { "486038", "10.0.1.3:11211" }, { "486271", "10.0.1.5:11211" },
      { "486504", "10.0.1.1:11211" }, { "486737", "10.0.1.5:11211" },
      { "486970", "10.0.1.3:11211" }, { "487203", "10.0.1.2:11211" },
      { "487436", "10.0.1.8:11211" }, { "487669", "10.0.1.3:11211" },
      { "487902", "10.0.1.7:11211" }, { "488135", "10.0.1.2:11211" },
      { "488368", "10.0.1.7:11211" }, { "488601", "10.0.1.8:11211" },
      { "488834", "10.0.1.8:11211" }, { "489067", "10.0.1.5:11211" },
      { "489300", "10.0.1.4:11211" }, { "489533", "10.0.1.5:11211" },
      { "489766", "10.0.1.5:11211" }, { "489999", "10.0.1.3:11211" },
      { "490232", "10.0.1.4:11211" }, { "490465", "10.0.1.2:11211" },
      { "490698", "10.0.1.1:11211" }, { "490931", "10.0.1.2:11211" },
      { "491164", "10.0.1.5:11211" }, { "491397", "10.0.1.5:11211" },
      { "491630", "10.0.1.2:11211" }, { "491863", "10.0.1.1:11211" },
      { "492096", "10.0.1.1:11211" }, { "492329", "10.0.1.5:11211" },
      { "492562", "10.0.1.7:11211" }, { "492795", "10.0.1.3:11211" },
      { "493028", "10.0.1.1:11211" }, { "493261", "10.0.1.2:11211" },
      { "493494", "10.0.1.3:11211" }, { "493727", "10.0.1.6:11211" },
      { "493960", "10.0.1.5:11211" }, { "494193", "10.0.1.6:11211" },
      { "494426", "10.0.1.6:11211" }, { "494659", "10.0.1.4:11211" },
      { "494892", "10.0.1.4:11211" }, { "495125", "10.0.1.1:11211" },
      { "495358", "10.0.1.3:11211" }, { "495591", "10.0.1.6:11211" },
      { "495824", "10.0.1.5:11211" }, { "496057", "10.0.1.7:11211" },
      { "496290", "10.0.1.5:11211" }, { "496523", "10.0.1.5:11211" },
      { "496756", "10.0.1.1:11211" }, { "496989", "10.0.1.3:11211" },
      { "497222", "10.0.1.8:11211" }, { "497455", "10.0.1.6:11211" },
      { "497688", "10.0.1.7:11211" }, { "497921", "10.0.1.5:11211" },
      { "498154", "10.0.1.4:11211" }, { "498387", "10.0.1.3:11211" },
      { "498620", "10.0.1.3:11211" }, { "498853", "10.0.1.3:11211" },
      { "499086", "10.0.1.1:11211" }, { "499319", "10.0.1.8:11211" },
      { "499552", "10.0.1.3:11211" }, { "499785", "10.0.1.2:11211" },
      { "500018", "10.0.1.5:11211" }, { "500251", "10.0.1.8:11211" },
      { "500484", "10.0.1.7:11211" }, { "500717", "10.0.1.4:11211" },
      { "500950", "10.0.1.5:11211" }, { "501183", "10.0.1.4:11211" },
      { "501416", "10.0.1.4:11211" }, { "501649", "10.0.1.2:11211" },
      { "501882", "10.0.1.1:11211" }, { "502115", "10.0.1.3:11211" },
      { "502348", "10.0.1.3:11211" }, { "502581", "10.0.1.8:11211" },
      { "502814", "10.0.1.1:11211" }, { "503047", "10.0.1.8:11211" },
      { "503280", "10.0.1.7:11211" }, { "503513", "10.0.1.1:11211" },
      { "503746", "10.0.1.1:11211" }, { "503979", "10.0.1.8:11211" },
      { "504212", "10.0.1.3:11211" }, { "504445", "10.0.1.2:11211" },
      { "504678", "10.0.1.2:11211" }, { "504911", "10.0.1.3:11211" },
      { "505144", "10.0.1.5:11211" }, { "505377", "10.0.1.6:11211" },
      { "505610", "10.0.1.1:11211" }, { "505843", "10.0.1.6:11211" },
      { "506076", "10.0.1.1:11211" }, { "506309", "10.0.1.2:11211" },
      { "506542", "10.0.1.3:11211" }, { "506775", "10.0.1.8:11211" },
      { "507008", "10.0.1.2:11211" }, { "507241", "10.0.1.3:11211" },
      { "507474", "10.0.1.6:11211" }, { "507707", "10.0.1.7:11211" },
      { "507940", "10.0.1.4:11211" }, { "508173", "10.0.1.5:11211" },
      { "508406", "10.0.1.1:11211" }, { "508639", "10.0.1.3:11211" },
      { "508872", "10.0.1.4:11211" }, { "509105", "10.0.1.7:11211" },
      { "509338", "10.0.1.2:11211" }, { "509571", "10.0.1.3:11211" },
      { "509804", "10.0.1.3:11211" }, { "510037", "10.0.1.7:11211" },
      { "510270", "10.0.1.4:11211" }, { "510503", "10.0.1.1:11211" },
      { "510736", "10.0.1.3:11211" }, { "510969", "10.0.1.6:11211" },
      { "511202", "10.0.1.2:11211" }, { "511435", "10.0.1.2:11211" },
      { "511668", "10.0.1.4:11211" }, { "511901", "10.0.1.6:11211" },
      { "512134", "10.0.1.2:11211" }, { "512367", "10.0.1.1:11211" },
      { "512600", "10.0.1.1:11211" }, { "512833", "10.0.1.3:11211" },
      { "513066", "10.0.1.3:11211" }, { "513299", "10.0.1.3:11211" },
      { "513532", "10.0.1.2:11211" }, { "513765", "10.0.1.8:11211" },
      { "513998", "10.0.1.7:11211" }, { "514231", "10.0.1.6:11211" },
      { "514464", "10.0.1.3:11211" }, { "514697", "10.0.1.3:11211" },
      { "514930", "10.0.1.6:11211" }, { "515163", "10.0.1.7:11211" },
      { "515396", "10.0.1.8:11211" }, { "515629", "10.0.1.5:11211" },
      { "515862", "10.0.1.7:11211" }, { "516095", "10.0.1.2:11211" },
      { "516328", "10.0.1.8:11211" }, { "516561", "10.0.1.3:11211" },
      { "516794", "10.0.1.5:11211" }, { "517027", "10.0.1.6:11211" },
      { "517260", "10.0.1.3:11211" }, { "517493", "10.0.1.5:11211" },
      { "517726", "10.0.1.8:11211" }, { "517959", "10.0.1.7:11211" },
      { "518192", "10.0.1.8:11211" }, { "518425", "10.0.1.8:11211" },
      { "518658", "10.0.1.8:11211" }, { "518891", "10.0.1.7:11211" },
      { "519124", "10.0.1.3:11211" }, { "519357", "10.0.1.6:11211" },
      { "519590", "10.0.1.5:11211" }, { "519823", "10.0.1.3:11211" },
      { "520056", "10.0.1.2:11211" }, { "520289", "10.0.1.8:11211" },
      { "520522", "10.0.1.5:11211" }, { "533337", "10.0.1.3:11211" },
      { "533570", "10.0.1.3:11211" }, { "533803", "10.0.1.2:11211" },
      { "534036", "10.0.1.4:11211" }, { "534269", "10.0.1.1:11211" },
      { "534502", "10.0.1.2:11211" }, { "534735", "10.0.1.4:11211" },
      { "534968", "10.0.1.8:11211" }, { "535201", "10.0.1.1:11211" },
      { "535434", "10.0.1.1:11211" }, { "535667", "10.0.1.3:11211" },
      { "535900", "10.0.1.7:11211" }, { "536133", "10.0.1.8:11211" },
      { "541026", "10.0.1.1:11211" }, { "541259", "10.0.1.2:11211" },
      { "541492", "10.0.1.6:11211" }, { "541725", "10.0.1.8:11211" },
      { "541958", "10.0.1.1:11211" }, { "542191", "10.0.1.6:11211" },
      { "542424", "10.0.1.7:11211" }, { "542657", "10.0.1.2:11211" },
      { "542890", "10.0.1.4:11211" }, { "543123", "10.0.1.5:11211" },
      { "543356", "10.0.1.3:11211" }, { "543589", "10.0.1.7:11211" },
      { "547084", "10.0.1.2:11211" }, { "547317", "10.0.1.3:11211" },
      { "547550", "10.0.1.7:11211" }, { "547783", "10.0.1.7:11211" },
      { "548016", "10.0.1.2:11211" }, { "548249", "10.0.1.5:11211" },
      { "548482", "10.0.1.7:11211" }, { "548715", "10.0.1.6:11211" },
      { "548948", "10.0.1.4:11211" }, { "549181", "10.0.1.3:11211" },
      { "549414", "10.0.1.3:11211" }, { "549647", "10.0.1.4:11211" },
      { "549880", "10.0.1.7:11211" }, { "550113", "10.0.1.6:11211" },
      { "550346", "10.0.1.4:11211" }, { "550579", "10.0.1.6:11211" },
      { "550812", "10.0.1.4:11211" }, { "551045", "10.0.1.5:11211" },
      { "551278", "10.0.1.6:11211" }, { "551511", "10.0.1.5:11211" },
      { "551744", "10.0.1.6:11211" }, { "551977", "10.0.1.8:11211" },
      { "552210", "10.0.1.2:11211" }, { "552443", "10.0.1.2:11211" },
      { "552676", "10.0.1.1:11211" }, { "552909", "10.0.1.4:11211" },
      { "553142", "10.0.1.7:11211" }, { "553375", "10.0.1.2:11211" },
      { "553608", "10.0.1.5:11211" }, { "553841", "10.0.1.5:11211" },
      { "554074", "10.0.1.5:11211" }, { "554307", "10.0.1.7:11211" },
      { "554540", "10.0.1.6:11211" }, { "554773", "10.0.1.3:11211" },
      { "555006", "10.0.1.3:11211" }, { "555239", "10.0.1.5:11211" },
      { "555472", "10.0.1.8:11211" }, { "555705", "10.0.1.8:11211" },
      { "555938", "10.0.1.6:11211" }, { "556171", "10.0.1.4:11211" },
      { "556404", "10.0.1.4:11211" }, { "556637", "10.0.1.8:11211" },
      { "556870", "10.0.1.5:11211" }, { "557103", "10.0.1.3:11211" },
      { "557336", "10.0.1.3:11211" }, { "557569", "10.0.1.8:11211" },
      { "557802", "10.0.1.1:11211" }, { "558035", "10.0.1.1:11211" },
      { "558268", "10.0.1.4:11211" }, { "558501", "10.0.1.3:11211" },
      { "558734", "10.0.1.6:11211" }, { "558967", "10.0.1.7:11211" },
      { "559200", "10.0.1.3:11211" }, { "559433", "10.0.1.1:11211" },
      { "559666", "10.0.1.4:11211" }, { "559899", "10.0.1.2:11211" },
      { "560132", "10.0.1.2:11211" }, { "560365", "10.0.1.6:11211" },
      { "560598", "10.0.1.8:11211" }, { "560831", "10.0.1.3:11211" },
      { "561064", "10.0.1.7:11211" }, { "561297", "10.0.1.1:11211" },
      { "561530", "10.0.1.7:11211" }, { "561763", "10.0.1.7:11211" },
      { "561996", "10.0.1.1:11211" }, { "562229", "10.0.1.8:11211" },
      { "562462", "10.0.1.6:11211" }, { "562695", "10.0.1.7:11211" },
      { "562928", "10.0.1.2:11211" }, { "563161", "10.0.1.8:11211" },
      { "563394", "10.0.1.8:11211" }, { "563627", "10.0.1.6:11211" },
      { "563860", "10.0.1.2:11211" }, { "564093", "10.0.1.5:11211" },
      { "564326", "10.0.1.4:11211" }, { "564559", "10.0.1.8:11211" },
      { "564792", "10.0.1.1:11211" }, { "565025", "10.0.1.3:11211" },
      { "565258", "10.0.1.7:11211" }, { "565491", "10.0.1.7:11211" },
      { "565724", "10.0.1.5:11211" }, { "565957", "10.0.1.4:11211" },
      { "566190", "10.0.1.4:11211" }, { "566423", "10.0.1.8:11211" },
      { "566656", "10.0.1.4:11211" }, { "566889", "10.0.1.3:11211" },
      { "567122", "10.0.1.1:11211" }, { "567355", "10.0.1.8:11211" },
      { "567588", "10.0.1.6:11211" }, { "567821", "10.0.1.3:11211" },
      { "568054", "10.0.1.5:11211" }, { "568287", "10.0.1.8:11211" },
      { "568520", "10.0.1.8:11211" }, { "568753", "10.0.1.4:11211" },
      { "568986", "10.0.1.5:11211" }, { "569219", "10.0.1.8:11211" },
      { "569452", "10.0.1.6:11211" }, { "569685", "10.0.1.1:11211" },
      { "569918", "10.0.1.3:11211" }, { "570151", "10.0.1.4:11211" },
      { "570384", "10.0.1.4:11211" }, { "570617", "10.0.1.8:11211" },
      { "570850", "10.0.1.5:11211" }, { "571083", "10.0.1.8:11211" },
      { "571316", "10.0.1.3:11211" }, { "571549", "10.0.1.3:11211" },
      { "571782", "10.0.1.2:11211" }, { "572015", "10.0.1.6:11211" },
      { "572248", "10.0.1.3:11211" }, { "572481", "10.0.1.7:11211" },
      { "572714", "10.0.1.6:11211" }, { "572947", "10.0.1.8:11211" },
      { "573180", "10.0.1.7:11211" }, { "573413", "10.0.1.1:11211" },
      { "573646", "10.0.1.1:11211" }, { "573879", "10.0.1.1:11211" },
      { "574112", "10.0.1.1:11211" }, { "574345", "10.0.1.1:11211" },
      { "574578", "10.0.1.6:11211" }, { "574811", "10.0.1.5:11211" },
      { "575044", "10.0.1.2:11211" }, { "575277", "10.0.1.1:11211" },
      { "575510", "10.0.1.1:11211" }, { "575743", "10.0.1.5:11211" },
      { "575976", "10.0.1.4:11211" }, { "576209", "10.0.1.6:11211" },
      { "576442", "10.0.1.5:11211" }, { "576675", "10.0.1.2:11211" },
      { "576908", "10.0.1.3:11211" }, { "577141", "10.0.1.7:11211" },
      { "577374", "10.0.1.5:11211" }, { "577607", "10.0.1.5:11211" },
      { "577840", "10.0.1.5:11211" }, { "578073", "10.0.1.1:11211" },
      { "578306", "10.0.1.2:11211" }, { "578539", "10.0.1.7:11211" },
      { "578772", "10.0.1.5:11211" }, { "579005", "10.0.1.6:11211" },
      { "579238", "10.0.1.4:11211" }, { "579471", "10.0.1.2:11211" },
      { "579704", "10.0.1.7:11211" }, { "579937", "10.0.1.6:11211" },
      { "580170", "10.0.1.4:11211" }, { "585063", "10.0.1.1:11211" },
      { "585296", "10.0.1.6:11211" }, { "585529", "10.0.1.4:11211" },
      { "585762", "10.0.1.4:11211" }, { "585995", "10.0.1.2:11211" },
      { "586228", "10.0.1.7:11211" }, { "586461", "10.0.1.8:11211" },
      { "586694", "10.0.1.3:11211" }, { "586927", "10.0.1.7:11211" },
      { "587160", "10.0.1.4:11211" }, { "587393", "10.0.1.8:11211" },
      { "587626", "10.0.1.7:11211" }, { "587859", "10.0.1.7:11211" },
      { "588092", "10.0.1.1:11211" }, { "588325", "10.0.1.8:11211" },
      { "588558", "10.0.1.3:11211" }, { "588791", "10.0.1.1:11211" },
      { "589024", "10.0.1.5:11211" }, { "589257", "10.0.1.4:11211" },
      { "589490", "10.0.1.1:11211" }, { "589723", "10.0.1.8:11211" },
      { "589956", "10.0.1.8:11211" }, { "590189", "10.0.1.1:11211" },
      { "590422", "10.0.1.8:11211" }, { "590655", "10.0.1.4:11211" },
      { "590888", "10.0.1.8:11211" }, { "591121", "10.0.1.1:11211" },
      { "591354", "10.0.1.6:11211" }, { "591587", "10.0.1.7:11211" },
      { "591820", "10.0.1.2:11211" }, { "592053", "10.0.1.7:11211" },
      { "592286", "10.0.1.8:11211" }, { "592519", "10.0.1.8:11211" },
      { "592752", "10.0.1.5:11211" }, { "592985", "10.0.1.5:11211" },
      { "593218", "10.0.1.5:11211" }, { "593451", "10.0.1.8:11211" },
      { "593684", "10.0.1.2:11211" }, { "593917", "10.0.1.3:11211" },
      { "594150", "10.0.1.5:11211" }, { "594383", "10.0.1.8:11211" },
      { "594616", "10.0.1.8:11211" }, { "594849", "10.0.1.2:11211" },
      { "595082", "10.0.1.3:11211" }, { "595315", "10.0.1.7:11211" },
      { "595548", "10.0.1.8:11211" }, { "595781", "10.0.1.2:11211" },
      { "596014", "10.0.1.5:11211" }, { "596247", "10.0.1.3:11211" },
      { "596480", "10.0.1.3:11211" }, { "596713", "10.0.1.3:11211" },
      { "596946", "10.0.1.8:11211" }, { "597179", "10.0.1.1:11211" },
      { "597412", "10.0.1.4:11211" }, { "597645", "10.0.1.3:11211" },
      { "597878", "10.0.1.4:11211" }, { "598111", "10.0.1.3:11211" },
      { "598344", "10.0.1.3:11211" }, { "598577", "10.0.1.7:11211" },
      { "598810", "10.0.1.3:11211" }, { "599043", "10.0.1.1:11211" },
      { "599276", "10.0.1.7:11211" }, { "599509", "10.0.1.5:11211" },
      { "599742", "10.0.1.7:11211" }, { "599975", "10.0.1.3:11211" },
      { "600208", "10.0.1.4:11211" }, { "600441", "10.0.1.1:11211" },
      { "600674", "10.0.1.8:11211" }, { "600907", "10.0.1.6:11211" },
      { "601140", "10.0.1.7:11211" }, { "601373", "10.0.1.2:11211" },
      { "601606", "10.0.1.5:11211" }, { "601839", "10.0.1.6:11211" },
      { "602072", "10.0.1.2:11211" }, { "602305", "10.0.1.3:11211" },
      { "602538", "10.0.1.3:11211" }, { "602771", "10.0.1.3:11211" },
      { "603004", "10.0.1.3:11211" }, { "603237", "10.0.1.8:11211" },
      { "603470", "10.0.1.5:11211" }, { "603703", "10.0.1.7:11211" },
      { "603936", "10.0.1.4:11211" }, { "604169", "10.0.1.7:11211" },
      { "604402", "10.0.1.2:11211" }, { "604635", "10.0.1.3:11211" },
      { "604868", "10.0.1.5:11211" }, { "605101", "10.0.1.5:11211" },
      { "614887", "10.0.1.2:11211" }, { "615120", "10.0.1.2:11211" },
      { "615353", "10.0.1.8:11211" }, { "615586", "10.0.1.6:11211" },
      { "615819", "10.0.1.5:11211" }, { "616052", "10.0.1.3:11211" },
      { "616285", "10.0.1.1:11211" }, { "616518", "10.0.1.8:11211" },
      { "616751", "10.0.1.5:11211" }, { "616984", "10.0.1.8:11211" },
      { "617217", "10.0.1.3:11211" }, { "617450", "10.0.1.4:11211" },
      { "617683", "10.0.1.1:11211" }, { "617916", "10.0.1.7:11211" },
      { "618149", "10.0.1.8:11211" }, { "618382", "10.0.1.7:11211" },
      { "618615", "10.0.1.7:11211" }, { "618848", "10.0.1.4:11211" },
      { "619081", "10.0.1.1:11211" }, { "619314", "10.0.1.8:11211" },
      { "619547", "10.0.1.3:11211" }, { "619780", "10.0.1.1:11211" },
      { "620013", "10.0.1.7:11211" }, { "620246", "10.0.1.5:11211" },
      { "620479", "10.0.1.3:11211" }, { "620712", "10.0.1.7:11211" },
      { "625139", "10.0.1.1:11211" }, { "625372", "10.0.1.8:11211" },
      { "625605", "10.0.1.7:11211" }, { "625838", "10.0.1.2:11211" },
      { "626071", "10.0.1.5:11211" }, { "626304", "10.0.1.3:11211" },
      { "626537", "10.0.1.5:11211" }, { "626770", "10.0.1.1:11211" },
      { "627003", "10.0.1.8:11211" }, { "627236", "10.0.1.4:11211" },
      { "627469", "10.0.1.4:11211" }, { "627702", "10.0.1.8:11211" },
      { "627935", "10.0.1.1:11211" }, { "628168", "10.0.1.6:11211" },
      { "628401", "10.0.1.4:11211" }, { "628634", "10.0.1.4:11211" },
      { "628867", "10.0.1.3:11211" }, { "629100", "10.0.1.5:11211" },
      { "629333", "10.0.1.4:11211" }, { "629566", "10.0.1.4:11211" },
      { "629799", "10.0.1.2:11211" }, { "630032", "10.0.1.5:11211" },
      { "630265", "10.0.1.8:11211" }, { "630498", "10.0.1.3:11211" },
      { "630731", "10.0.1.6:11211" }, { "630964", "10.0.1.6:11211" },
      { "631197", "10.0.1.2:11211" }, { "631430", "10.0.1.3:11211" },
      { "631663", "10.0.1.7:11211" }, { "631896", "10.0.1.4:11211" },
      { "632129", "10.0.1.2:11211" }, { "632362", "10.0.1.7:11211" },
      { "632595", "10.0.1.6:11211" }, { "632828", "10.0.1.3:11211" },
      { "633061", "10.0.1.8:11211" }, { "633294", "10.0.1.5:11211" },
      { "633527", "10.0.1.3:11211" }, { "633760", "10.0.1.6:11211" },
      { "633993", "10.0.1.7:11211" }, { "634226", "10.0.1.2:11211" },
      { "634459", "10.0.1.1:11211" }, { "634692", "10.0.1.2:11211" },
      { "634925", "10.0.1.6:11211" }, { "635158", "10.0.1.4:11211" },
      { "635391", "10.0.1.4:11211" }, { "635624", "10.0.1.4:11211" },
      { "635857", "10.0.1.5:11211" }, { "636090", "10.0.1.2:11211" },
      { "636323", "10.0.1.3:11211" }, { "636556", "10.0.1.1:11211" },
      { "636789", "10.0.1.4:11211" }, { "637022", "10.0.1.1:11211" },
      { "637255", "10.0.1.4:11211" }, { "637488", "10.0.1.3:11211" },
      { "637721", "10.0.1.3:11211" }, { "637954", "10.0.1.2:11211" },
      { "638187", "10.0.1.2:11211" }, { "638420", "10.0.1.5:11211" },
      { "638653", "10.0.1.5:11211" }, { "638886", "10.0.1.2:11211" },
      { "639119", "10.0.1.6:11211" }, { "639352", "10.0.1.6:11211" },
      { "639585", "10.0.1.2:11211" }, { "639818", "10.0.1.7:11211" },
      { "640051", "10.0.1.3:11211" }, { "640284", "10.0.1.3:11211" },
      { "640517", "10.0.1.7:11211" }, { "640750", "10.0.1.4:11211" },
      { "640983", "10.0.1.3:11211" }, { "641216", "10.0.1.2:11211" },
      { "641449", "10.0.1.1:11211" }, { "641682", "10.0.1.4:11211" },
      { "641915", "10.0.1.5:11211" }, { "642148", "10.0.1.1:11211" },
      { "642381", "10.0.1.2:11211" }, { "642614", "10.0.1.7:11211" },
      { "642847", "10.0.1.7:11211" }, { "643080", "10.0.1.3:11211" },
      { "643313", "10.0.1.6:11211" }, { "643546", "10.0.1.4:11211" },
      { "643779", "10.0.1.3:11211" }, { "644012", "10.0.1.5:11211" },
      { "644245", "10.0.1.3:11211" }, { "644478", "10.0.1.8:11211" },
      { "644711", "10.0.1.8:11211" }, { "644944", "10.0.1.2:11211" },
      { "645177", "10.0.1.8:11211" }, { "645410", "10.0.1.6:11211" },
      { "645643", "10.0.1.2:11211" }, { "645876", "10.0.1.4:11211" },
      { "646109", "10.0.1.4:11211" }, { "646342", "10.0.1.1:11211" },
      { "646575", "10.0.1.3:11211" }, { "646808", "10.0.1.3:11211" },
      { "647041", "10.0.1.4:11211" }, { "647274", "10.0.1.5:11211" },
      { "647507", "10.0.1.7:11211" }, { "647740", "10.0.1.1:11211" },
      { "647973", "10.0.1.7:11211" }, { "648206", "10.0.1.7:11211" },
      { "648439", "10.0.1.8:11211" }, { "648672", "10.0.1.4:11211" },
      { "648905", "10.0.1.4:11211" }, { "649138", "10.0.1.5:11211" },
      { "649371", "10.0.1.6:11211" }, { "649604", "10.0.1.3:11211" },
      { "649837", "10.0.1.7:11211" }, { "650070", "10.0.1.5:11211" },
      { "650303", "10.0.1.4:11211" }, { "650536", "10.0.1.8:11211" },
      { "650769", "10.0.1.8:11211" }, { "651002", "10.0.1.8:11211" },
      { "651235", "10.0.1.4:11211" }, { "651468", "10.0.1.1:11211" },
      { "651701", "10.0.1.3:11211" }, { "651934", "10.0.1.4:11211" },
      { "652167", "10.0.1.4:11211" }, { "652400", "10.0.1.7:11211" },
      { "652633", "10.0.1.4:11211" }, { "652866", "10.0.1.3:11211" },
      { "653099", "10.0.1.1:11211" }, { "653332", "10.0.1.8:11211" },
      { "653565", "10.0.1.2:11211" }, { "653798", "10.0.1.2:11211" },
      { "654031", "10.0.1.6:11211" }, { "654264", "10.0.1.2:11211" },
      { "654497", "10.0.1.4:11211" }, { "654730", "10.0.1.7:11211" },
      { "654963", "10.0.1.8:11211" }, { "655196", "10.0.1.1:11211" },
      { "655429", "10.0.1.8:11211" }, { "655662", "10.0.1.5:11211" },
      { "655895", "10.0.1.7:11211" }, { "656128", "10.0.1.3:11211" },
      { "656361", "10.0.1.8:11211" }, { "656594", "10.0.1.5:11211" },
      { "656827", "10.0.1.6:11211" }, { "657060", "10.0.1.3:11211" },
      { "657293", "10.0.1.2:11211" }, { "657526", "10.0.1.5:11211" },
      { "657759", "10.0.1.2:11211" }, { "657992", "10.0.1.4:11211" },
      { "658225", "10.0.1.5:11211" }, { "658458", "10.0.1.8:11211" },
      { "658691", "10.0.1.1:11211" }, { "658924", "10.0.1.4:11211" },
      { "659157", "10.0.1.2:11211" }, { "659390", "10.0.1.2:11211" },
      { "659623", "10.0.1.2:11211" }, { "659856", "10.0.1.3:11211" },
      { "660089", "10.0.1.8:11211" }, { "660322", "10.0.1.7:11211" },
      { "660555", "10.0.1.4:11211" }, { "660788", "10.0.1.7:11211" },
      { "661021", "10.0.1.1:11211" }, { "661254", "10.0.1.7:11211" },
      { "661487", "10.0.1.2:11211" }, { "661720", "10.0.1.5:11211" },
      { "661953", "10.0.1.7:11211" }, { "662186", "10.0.1.1:11211" },
      { "662419", "10.0.1.1:11211" }, { "662652", "10.0.1.3:11211" },
      { "662885", "10.0.1.7:11211" }, { "677564", "10.0.1.5:11211" },
      { "677797", "10.0.1.7:11211" }, { "678030", "10.0.1.6:11211" },
      { "678263", "10.0.1.8:11211" }, { "678496", "10.0.1.8:11211" },
      { "678729", "10.0.1.6:11211" }, { "678962", "10.0.1.1:11211" },
      { "679195", "10.0.1.7:11211" }, { "679428", "10.0.1.6:11211" },
      { "679661", "10.0.1.5:11211" }, { "679894", "10.0.1.8:11211" },
      { "680127", "10.0.1.5:11211" }, { "680360", "10.0.1.1:11211" },
      { "680593", "10.0.1.8:11211" }, { "680826", "10.0.1.7:11211" },
      { "681059", "10.0.1.5:11211" }, { "681292", "10.0.1.7:11211" },
      { "681525", "10.0.1.3:11211" }, { "681758", "10.0.1.3:11211" },
      { "685952", "10.0.1.4:11211" }, { "686185", "10.0.1.6:11211" },
      { "686418", "10.0.1.5:11211" }, { "686651", "10.0.1.3:11211" },
      { "686884", "10.0.1.8:11211" }, { "687117", "10.0.1.6:11211" },
      { "687350", "10.0.1.4:11211" }, { "687583", "10.0.1.8:11211" },
      { "687816", "10.0.1.7:11211" }, { "688049", "10.0.1.7:11211" },
      { "688282", "10.0.1.5:11211" }, { "688515", "10.0.1.6:11211" },
      { "688748", "10.0.1.4:11211" }, { "688981", "10.0.1.1:11211" },
      { "689214", "10.0.1.5:11211" }, { "689447", "10.0.1.3:11211" },
      { "689680", "10.0.1.8:11211" }, { "689913", "10.0.1.8:11211" },
      { "690146", "10.0.1.6:11211" }, { "690379", "10.0.1.8:11211" },
      { "690612", "10.0.1.2:11211" }, { "690845", "10.0.1.6:11211" },
      { "691078", "10.0.1.2:11211" }, { "691311", "10.0.1.7:11211" },
      { "691544", "10.0.1.8:11211" }, { "691777", "10.0.1.5:11211" },
      { "692010", "10.0.1.7:11211" }, { "692243", "10.0.1.4:11211" },
      { "692476", "10.0.1.3:11211" }, { "692709", "10.0.1.3:11211" },
      { "692942", "10.0.1.1:11211" }, { "693175", "10.0.1.5:11211" },
      { "693408", "10.0.1.5:11211" }, { "693641", "10.0.1.6:11211" },
      { "693874", "10.0.1.3:11211" }, { "694107", "10.0.1.3:11211" },
      { "694340", "10.0.1.5:11211" }, { "694573", "10.0.1.4:11211" },
      { "694806", "10.0.1.6:11211" }, { "695039", "10.0.1.1:11211" },
      { "695272", "10.0.1.6:11211" }, { "695505", "10.0.1.3:11211" },
      { "695738", "10.0.1.6:11211" }, { "695971", "10.0.1.8:11211" },
      { "696204", "10.0.1.2:11211" }, { "696437", "10.0.1.1:11211" },
      { "696670", "10.0.1.6:11211" }, { "696903", "10.0.1.7:11211" },
      { "697136", "10.0.1.2:11211" }, { "697369", "10.0.1.6:11211" },
      { "697602", "10.0.1.2:11211" }, { "697835", "10.0.1.4:11211" },
      { "698068", "10.0.1.8:11211" }, { "698301", "10.0.1.5:11211" },
      { "698534", "10.0.1.4:11211" }, { "698767", "10.0.1.6:11211" },
      { "699000", "10.0.1.7:11211" }, { "699233", "10.0.1.4:11211" },
      { "699466", "10.0.1.5:11211" }, { "699699", "10.0.1.3:11211" },
      { "699932", "10.0.1.2:11211" }, { "700165", "10.0.1.6:11211" },
      { "700398", "10.0.1.1:11211" }, { "700631", "10.0.1.1:11211" },
      { "700864", "10.0.1.3:11211" }, { "701097", "10.0.1.4:11211" },
      { "701330", "10.0.1.7:11211" }, { "701563", "10.0.1.6:11211" },
      { "701796", "10.0.1.8:11211" }, { "702029", "10.0.1.4:11211" },
      { "702262", "10.0.1.4:11211" }, { "702495", "10.0.1.5:11211" },
      { "702728", "10.0.1.7:11211" }, { "702961", "10.0.1.4:11211" },
      { "703194", "10.0.1.5:11211" }, { "703427", "10.0.1.7:11211" },
      { "703660", "10.0.1.1:11211" }, { "703893", "10.0.1.6:11211" },
      { "704126", "10.0.1.4:11211" }, { "704359", "10.0.1.1:11211" },
      { "704592", "10.0.1.4:11211" }, { "704825", "10.0.1.2:11211" },
      { "705058", "10.0.1.8:11211" }, { "705291", "10.0.1.6:11211" },
      { "705524", "10.0.1.3:11211" }, { "705757", "10.0.1.7:11211" },
      { "705990", "10.0.1.7:11211" }, { "706223", "10.0.1.8:11211" },
      { "706456", "10.0.1.7:11211" }, { "706689", "10.0.1.8:11211" },
      { "706922", "10.0.1.7:11211" }, { "707155", "10.0.1.8:11211" },
      { "707388", "10.0.1.4:11211" }, { "707621", "10.0.1.8:11211" },
      { "707854", "10.0.1.8:11211" }, { "708087", "10.0.1.7:11211" },
      { "708320", "10.0.1.5:11211" }, { "708553", "10.0.1.6:11211" },
      { "708786", "10.0.1.3:11211" }, { "709019", "10.0.1.7:11211" },
      { "709252", "10.0.1.4:11211" }, { "709485", "10.0.1.8:11211" },
      { "709718", "10.0.1.1:11211" }, { "709951", "10.0.1.3:11211" },
      { "710184", "10.0.1.1:11211" }, { "710417", "10.0.1.6:11211" },
      { "710650", "10.0.1.1:11211" }, { "710883", "10.0.1.7:11211" },
      { "711116", "10.0.1.4:11211" }, { "711349", "10.0.1.7:11211" },
      { "711582", "10.0.1.1:11211" }, { "711815", "10.0.1.7:11211" },
      { "712048", "10.0.1.6:11211" }, { "712281", "10.0.1.7:11211" },
      { "712514", "10.0.1.2:11211" }, { "712747", "10.0.1.3:11211" },
      { "712980", "10.0.1.1:11211" }, { "713213", "10.0.1.8:11211" },
      { "713446", "10.0.1.2:11211" }, { "713679", "10.0.1.8:11211" },
      { "713912", "10.0.1.3:11211" }, { "714145", "10.0.1.3:11211" },
      { "714378", "10.0.1.6:11211" }, { "714611", "10.0.1.8:11211" },
      { "714844", "10.0.1.3:11211" }, { "715077", "10.0.1.8:11211" },
      { "715310", "10.0.1.3:11211" }, { "715543", "10.0.1.7:11211" },
      { "715776", "10.0.1.2:11211" }, { "716009", "10.0.1.5:11211" },
      { "716242", "10.0.1.7:11211" }, { "716475", "10.0.1.8:11211" },
      { "716708", "10.0.1.7:11211" }, { "716941", "10.0.1.6:11211" },
      { "717174", "10.0.1.3:11211" }, { "717407", "10.0.1.2:11211" },
      { "717640", "10.0.1.3:11211" }, { "717873", "10.0.1.3:11211" },
      { "718106", "10.0.1.2:11211" }, { "718339", "10.0.1.8:11211" },
      { "718572", "10.0.1.5:11211" }, { "718805", "10.0.1.5:11211" },
      { "719038", "10.0.1.8:11211" }, { "719271", "10.0.1.5:11211" },
      { "719504", "10.0.1.3:11211" }, { "719737", "10.0.1.5:11211" },
      { "719970", "10.0.1.6:11211" }, { "720203", "10.0.1.7:11211" },
      { "720436", "10.0.1.2:11211" }, { "720669", "10.0.1.7:11211" },
      { "720902", "10.0.1.7:11211" }, { "721135", "10.0.1.2:11211" },
      { "721368", "10.0.1.1:11211" }, { "721601", "10.0.1.2:11211" },
      { "721834", "10.0.1.3:11211" }, { "722067", "10.0.1.5:11211" },
      { "722300", "10.0.1.8:11211" }, { "722533", "10.0.1.1:11211" },
      { "722766", "10.0.1.4:11211" }, { "722999", "10.0.1.8:11211" },
      { "723232", "10.0.1.8:11211" }, { "723465", "10.0.1.8:11211" },
      { "723698", "10.0.1.2:11211" }, { "723931", "10.0.1.6:11211" },
      { "724164", "10.0.1.5:11211" }, { "724397", "10.0.1.8:11211" },
      { "724630", "10.0.1.5:11211" }, { "724863", "10.0.1.4:11211" },
      { "725096", "10.0.1.4:11211" }, { "725329", "10.0.1.8:11211" },
      { "725562", "10.0.1.1:11211" }, { "729290", "10.0.1.7:11211" },
      { "729523", "10.0.1.1:11211" }, { "729756", "10.0.1.1:11211" },
      { "729989", "10.0.1.5:11211" }, { "730222", "10.0.1.3:11211" },
      { "730455", "10.0.1.3:11211" }, { "730688", "10.0.1.6:11211" },
      { "730921", "10.0.1.1:11211" }, { "731154", "10.0.1.4:11211" },
      { "731387", "10.0.1.1:11211" }, { "731620", "10.0.1.7:11211" },
      { "731853", "10.0.1.3:11211" }, { "732086", "10.0.1.7:11211" },
      { "732319", "10.0.1.7:11211" }, { "732552", "10.0.1.6:11211" },
      { "732785", "10.0.1.2:11211" }, { "733018", "10.0.1.5:11211" },
      { "733251", "10.0.1.8:11211" }, { "733484", "10.0.1.3:11211" },
      { "733717", "10.0.1.5:11211" }, { "733950", "10.0.1.5:11211" },
      { "734183", "10.0.1.1:11211" }, { "734416", "10.0.1.7:11211" },
      { "734649", "10.0.1.4:11211" }, { "734882", "10.0.1.8:11211" },
      { "735115", "10.0.1.4:11211" }, { "735348", "10.0.1.1:11211" },
      { "735581", "10.0.1.7:11211" }, { "735814", "10.0.1.2:11211" },
      { "736047", "10.0.1.1:11211" }, { "736280", "10.0.1.3:11211" },
      { "736513", "10.0.1.3:11211" }, { "736746", "10.0.1.6:11211" },
      { "736979", "10.0.1.3:11211" }, { "737212", "10.0.1.6:11211" },
      { "737445", "10.0.1.4:11211" }, { "737678", "10.0.1.4:11211" },
      { "737911", "10.0.1.4:11211" }, { "740474", "10.0.1.7:11211" },
      { "740707", "10.0.1.2:11211" }, { "740940", "10.0.1.1:11211" },
      { "741173", "10.0.1.2:11211" }, { "741406", "10.0.1.5:11211" },
      { "741639", "10.0.1.8:11211" }, { "741872", "10.0.1.5:11211" },
      { "742105", "10.0.1.6:11211" }, { "742338", "10.0.1.7:11211" },
      { "742571", "10.0.1.1:11211" }, { "742804", "10.0.1.7:11211" },
      { "743037", "10.0.1.7:11211" }, { "743270", "10.0.1.1:11211" },
      { "743503", "10.0.1.1:11211" }, { "743736", "10.0.1.5:11211" },
      { "743969", "10.0.1.6:11211" }, { "744202", "10.0.1.7:11211" },
      { "744435", "10.0.1.2:11211" }, { "744668", "10.0.1.2:11211" },
      { "744901", "10.0.1.5:11211" }, { "745134", "10.0.1.1:11211" },
      { "745367", "10.0.1.7:11211" }, { "745600", "10.0.1.7:11211" },
      { "745833", "10.0.1.7:11211" }, { "746066", "10.0.1.6:11211" },
      { "746299", "10.0.1.5:11211" }, { "746532", "10.0.1.3:11211" },
      { "746765", "10.0.1.6:11211" }, { "746998", "10.0.1.3:11211" },
      { "747231", "10.0.1.7:11211" }, { "747464", "10.0.1.1:11211" },
      { "747697", "10.0.1.6:11211" }, { "747930", "10.0.1.3:11211" },
      { "748163", "10.0.1.6:11211" }, { "748396", "10.0.1.7:11211" },
      { "748629", "10.0.1.6:11211" }, { "748862", "10.0.1.2:11211" },
      { "749095", "10.0.1.4:11211" }, { "749328", "10.0.1.7:11211" },
      { "749561", "10.0.1.4:11211" }, { "749794", "10.0.1.5:11211" },
      { "750027", "10.0.1.3:11211" }, { "750260", "10.0.1.6:11211" },
      { "750493", "10.0.1.5:11211" }, { "750726", "10.0.1.4:11211" },
      { "750959", "10.0.1.1:11211" }, { "751192", "10.0.1.1:11211" },
      { "751425", "10.0.1.6:11211" }, { "751658", "10.0.1.3:11211" },
      { "751891", "10.0.1.6:11211" }, { "752124", "10.0.1.6:11211" },
      { "752357", "10.0.1.5:11211" }, { "752590", "10.0.1.3:11211" },
      { "752823", "10.0.1.2:11211" }, { "753056", "10.0.1.1:11211" },
      { "753289", "10.0.1.7:11211" }, { "753522", "10.0.1.8:11211" },
      { "753755", "10.0.1.3:11211" }, { "753988", "10.0.1.2:11211" },
      { "754221", "10.0.1.8:11211" }, { "754454", "10.0.1.1:11211" },
      { "754687", "10.0.1.7:11211" }, { "754920", "10.0.1.6:11211" },
      { "755153", "10.0.1.2:11211" }, { "755386", "10.0.1.1:11211" },
      { "755619", "10.0.1.7:11211" }, { "755852", "10.0.1.3:11211" },
      { "756085", "10.0.1.5:11211" }, { "756318", "10.0.1.6:11211" },
      { "756551", "10.0.1.3:11211" }, { "756784", "10.0.1.7:11211" },
      { "757017", "10.0.1.7:11211" }, { "757250", "10.0.1.2:11211" },
      { "757483", "10.0.1.2:11211" }, { "757716", "10.0.1.3:11211" },
      { "757949", "10.0.1.2:11211" }, { "758182", "10.0.1.3:11211" },
      { "758415", "10.0.1.4:11211" }, { "758648", "10.0.1.6:11211" },
      { "758881", "10.0.1.6:11211" }, { "759114", "10.0.1.5:11211" },
      { "759347", "10.0.1.8:11211" }, { "759580", "10.0.1.8:11211" },
      { "759813", "10.0.1.1:11211" }, { "760046", "10.0.1.1:11211" },
      { "760279", "10.0.1.3:11211" }, { "760512", "10.0.1.6:11211" },
      { "760745", "10.0.1.8:11211" }, { "760978", "10.0.1.6:11211" },
      { "761211", "10.0.1.4:11211" }, { "761444", "10.0.1.7:11211" },
      { "761677", "10.0.1.1:11211" }, { "761910", "10.0.1.4:11211" },
      { "762143", "10.0.1.7:11211" }, { "762376", "10.0.1.7:11211" },
      { "762609", "10.0.1.2:11211" }, { "762842", "10.0.1.6:11211" },
      { "763075", "10.0.1.7:11211" }, { "763308", "10.0.1.3:11211" },
      { "763541", "10.0.1.7:11211" }, { "763774", "10.0.1.5:11211" },
      { "764007", "10.0.1.3:11211" }, { "764240", "10.0.1.6:11211" },
      { "764473", "10.0.1.4:11211" }, { "764706", "10.0.1.3:11211" },
      { "764939", "10.0.1.2:11211" }, { "765172", "10.0.1.8:11211" },
      { "765405", "10.0.1.2:11211" }, { "765638", "10.0.1.6:11211" },
      { "765871", "10.0.1.1:11211" }, { "766104", "10.0.1.1:11211" },
      { "766337", "10.0.1.5:11211" }, { "766570", "10.0.1.4:11211" },
      { "766803", "10.0.1.5:11211" }, { "767036", "10.0.1.6:11211" },
      { "767269", "10.0.1.3:11211" }, { "767502", "10.0.1.5:11211" },
      { "767735", "10.0.1.2:11211" }, { "767968", "10.0.1.3:11211" },
      { "768201", "10.0.1.7:11211" }, { "777521", "10.0.1.2:11211" },
      { "777754", "10.0.1.4:11211" }, { "777987", "10.0.1.2:11211" },
      { "778220", "10.0.1.3:11211" }, { "778453", "10.0.1.6:11211" },
      { "778686", "10.0.1.8:11211" }, { "778919", "10.0.1.1:11211" },
      { "779152", "10.0.1.1:11211" }, { "779385", "10.0.1.3:11211" },
      { "779618", "10.0.1.5:11211" }, { "779851", "10.0.1.8:11211" },
      { "780084", "10.0.1.7:11211" }, { "780317", "10.0.1.2:11211" },
      { "780550", "10.0.1.1:11211" }, { "780783", "10.0.1.2:11211" },
      { "781016", "10.0.1.3:11211" }, { "781249", "10.0.1.6:11211" },
      { "781482", "10.0.1.1:11211" }, { "781715", "10.0.1.7:11211" },
      { "781948", "10.0.1.3:11211" }, { "782181", "10.0.1.6:11211" },
      { "782414", "10.0.1.2:11211" }, { "782647", "10.0.1.5:11211" },
      { "782880", "10.0.1.5:11211" }, { "783113", "10.0.1.4:11211" },
      { "783346", "10.0.1.8:11211" }, { "783579", "10.0.1.8:11211" },
      { "783812", "10.0.1.1:11211" }, { "784045", "10.0.1.5:11211" },
      { "784278", "10.0.1.1:11211" }, { "784511", "10.0.1.8:11211" },
      { "784744", "10.0.1.8:11211" }, { "784977", "10.0.1.1:11211" },
      { "785210", "10.0.1.7:11211" }, { "785443", "10.0.1.2:11211" },
      { "785676", "10.0.1.8:11211" }, { "785909", "10.0.1.3:11211" },
      { "786142", "10.0.1.8:11211" }, { "786375", "10.0.1.8:11211" },
      { "786608", "10.0.1.2:11211" }, { "786841", "10.0.1.1:11211" },
      { "787074", "10.0.1.8:11211" }, { "787307", "10.0.1.7:11211" },
      { "787540", "10.0.1.2:11211" }, { "787773", "10.0.1.3:11211" },
      { "788006", "10.0.1.8:11211" }, { "788239", "10.0.1.1:11211" },
      { "788472", "10.0.1.2:11211" }, { "788705", "10.0.1.6:11211" },
      { "788938", "10.0.1.7:11211" }, { "789171", "10.0.1.7:11211" },
      { "789404", "10.0.1.2:11211" }, { "789637", "10.0.1.8:11211" },
      { "789870", "10.0.1.3:11211" }, { "790103", "10.0.1.8:11211" },
      { "790336", "10.0.1.6:11211" }, { "790569", "10.0.1.5:11211" },
      { "790802", "10.0.1.6:11211" }, { "791035", "10.0.1.7:11211" },
      { "791268", "10.0.1.7:11211" }, { "791501", "10.0.1.7:11211" },
      { "791734", "10.0.1.8:11211" }, { "791967", "10.0.1.6:11211" },
      { "792200", "10.0.1.3:11211" }, { "792433", "10.0.1.6:11211" },
      { "792666", "10.0.1.7:11211" }, { "792899", "10.0.1.2:11211" },
      { "793132", "10.0.1.1:11211" }, { "793365", "10.0.1.5:11211" },
      { "793598", "10.0.1.2:11211" }, { "793831", "10.0.1.6:11211" },
      { "794064", "10.0.1.3:11211" }, { "794297", "10.0.1.1:11211" },
      { "794530", "10.0.1.1:11211" }, { "794763", "10.0.1.7:11211" },
      { "794996", "10.0.1.6:11211" }, { "795229", "10.0.1.5:11211" },
      { "795462", "10.0.1.1:11211" }, { "795695", "10.0.1.8:11211" },
      { "795928", "10.0.1.4:11211" }, { "796161", "10.0.1.3:11211" },
      { "796394", "10.0.1.7:11211" }, { "796627", "10.0.1.5:11211" },
      { "796860", "10.0.1.8:11211" }, { "797093", "10.0.1.7:11211" },
      { "797326", "10.0.1.3:11211" }, { "797559", "10.0.1.3:11211" },
      { "797792", "10.0.1.8:11211" }, { "798025", "10.0.1.4:11211" },
      { "798258", "10.0.1.3:11211" }, { "798491", "10.0.1.3:11211" },
      { "798724", "10.0.1.8:11211" }, { "798957", "10.0.1.2:11211" },
      { "799190", "10.0.1.5:11211" }, { "799423", "10.0.1.2:11211" },
      { "799656", "10.0.1.6:11211" }, { "799889", "10.0.1.5:11211" },
      { "800122", "10.0.1.5:11211" }, { "800355", "10.0.1.3:11211" },
      { "800588", "10.0.1.6:11211" }, { "800821", "10.0.1.5:11211" },
      { "801054", "10.0.1.3:11211" }, { "801287", "10.0.1.2:11211" },
      { "801520", "10.0.1.2:11211" }, { "801753", "10.0.1.8:11211" },
      { "801986", "10.0.1.2:11211" }, { "802219", "10.0.1.1:11211" },
      { "802452", "10.0.1.2:11211" }, { "802685", "10.0.1.7:11211" },
      { "802918", "10.0.1.8:11211" }, { "803151", "10.0.1.6:11211" },
      { "803384", "10.0.1.7:11211" }, { "803617", "10.0.1.7:11211" },
      { "803850", "10.0.1.3:11211" }, { "804083", "10.0.1.8:11211" },
      { "804316", "10.0.1.8:11211" }, { "804549", "10.0.1.6:11211" },
      { "804782", "10.0.1.4:11211" }, { "805015", "10.0.1.7:11211" },
      { "805248", "10.0.1.5:11211" }, { "805481", "10.0.1.2:11211" },
      { "805714", "10.0.1.7:11211" }, { "805947", "10.0.1.6:11211" },
      { "806180", "10.0.1.2:11211" }, { "806413", "10.0.1.3:11211" },
      { "806646", "10.0.1.2:11211" }, { "806879", "10.0.1.5:11211" },
      { "807112", "10.0.1.8:11211" }, { "807345", "10.0.1.1:11211" },
      { "807578", "10.0.1.7:11211" }, { "807811", "10.0.1.5:11211" },
      { "808044", "10.0.1.2:11211" }, { "808277", "10.0.1.7:11211" },
      { "808510", "10.0.1.5:11211" }, { "808743", "10.0.1.8:11211" },
      { "811772", "10.0.1.5:11211" }, { "812005", "10.0.1.7:11211" },
      { "812238", "10.0.1.1:11211" }, { "812471", "10.0.1.2:11211" },
      { "815500", "10.0.1.5:11211" }, { "815733", "10.0.1.6:11211" },
      { "815966", "10.0.1.8:11211" }, { "816199", "10.0.1.8:11211" },
      { "816432", "10.0.1.2:11211" }, { "816665", "10.0.1.4:11211" },
      { "816898", "10.0.1.1:11211" }, { "817131", "10.0.1.1:11211" },
      { "817364", "10.0.1.7:11211" }, { "817597", "10.0.1.3:11211" },
      { "817830", "10.0.1.2:11211" }, { "818063", "10.0.1.7:11211" },
      { "818296", "10.0.1.6:11211" }, { "818529", "10.0.1.4:11211" },
      { "818762", "10.0.1.3:11211" }, { "818995", "10.0.1.2:11211" },
      { "819228", "10.0.1.4:11211" }, { "819461", "10.0.1.7:11211" },
      { "819694", "10.0.1.5:11211" }, { "819927", "10.0.1.6:11211" },
      { "820160", "10.0.1.5:11211" }, { "820393", "10.0.1.2:11211" },
      { "820626", "10.0.1.1:11211" }, { "820859", "10.0.1.6:11211" },
      { "821092", "10.0.1.8:11211" }, { "821325", "10.0.1.2:11211" },
      { "821558", "10.0.1.6:11211" }, { "821791", "10.0.1.2:11211" },
      { "822024", "10.0.1.1:11211" }, { "822257", "10.0.1.3:11211" },
      { "822490", "10.0.1.1:11211" }, { "822723", "10.0.1.6:11211" },
      { "822956", "10.0.1.1:11211" }, { "823189", "10.0.1.7:11211" },
      { "823422", "10.0.1.2:11211" }, { "823655", "10.0.1.3:11211" },
      { "823888", "10.0.1.7:11211" }, { "824121", "10.0.1.2:11211" },
      { "824354", "10.0.1.6:11211" }, { "824587", "10.0.1.7:11211" },
      { "824820", "10.0.1.8:11211" }, { "825053", "10.0.1.5:11211" },
      { "825286", "10.0.1.4:11211" }, { "825519", "10.0.1.8:11211" },
      { "825752", "10.0.1.8:11211" }, { "825985", "10.0.1.8:11211" },
      { "826218", "10.0.1.8:11211" }, { "826451", "10.0.1.4:11211" },
      { "826684", "10.0.1.7:11211" }, { "826917", "10.0.1.5:11211" },
      { "827150", "10.0.1.3:11211" }, { "827383", "10.0.1.1:11211" },
      { "827616", "10.0.1.8:11211" }, { "827849", "10.0.1.1:11211" },
      { "828082", "10.0.1.7:11211" }, { "828315", "10.0.1.7:11211" },
      { "828548", "10.0.1.4:11211" }, { "828781", "10.0.1.7:11211" },
      { "829014", "10.0.1.5:11211" }, { "829247", "10.0.1.7:11211" },
      { "829480", "10.0.1.2:11211" }, { "829713", "10.0.1.4:11211" },
      { "829946", "10.0.1.2:11211" }, { "830179", "10.0.1.1:11211" },
      { "830412", "10.0.1.3:11211" }, { "830645", "10.0.1.6:11211" },
      { "830878", "10.0.1.7:11211" }, { "831111", "10.0.1.1:11211" },
      { "831344", "10.0.1.7:11211" }, { "831577", "10.0.1.7:11211" },
      { "831810", "10.0.1.1:11211" }, { "832043", "10.0.1.8:11211" },
      { "832276", "10.0.1.3:11211" }, { "832509", "10.0.1.3:11211" },
      { "832742", "10.0.1.6:11211" }, { "832975", "10.0.1.4:11211" },
      { "833208", "10.0.1.1:11211" }, { "833441", "10.0.1.4:11211" },
      { "833674", "10.0.1.7:11211" }, { "833907", "10.0.1.3:11211" },
      { "834140", "10.0.1.4:11211" }, { "834373", "10.0.1.3:11211" },
      { "834606", "10.0.1.7:11211" }, { "834839", "10.0.1.3:11211" },
      { "835072", "10.0.1.4:11211" }, { "835305", "10.0.1.5:11211" },
      { "835538", "10.0.1.3:11211" }, { "835771", "10.0.1.3:11211" },
      { "836004", "10.0.1.6:11211" }, { "836237", "10.0.1.7:11211" },
      { "836470", "10.0.1.1:11211" }, { "836703", "10.0.1.7:11211" },
      { "836936", "10.0.1.2:11211" }, { "837169", "10.0.1.4:11211" },
      { "837402", "10.0.1.2:11211" }, { "837635", "10.0.1.6:11211" },
      { "837868", "10.0.1.8:11211" }, { "838101", "10.0.1.2:11211" },
      { "838334", "10.0.1.4:11211" }, { "838567", "10.0.1.4:11211" },
      { "838800", "10.0.1.7:11211" }, { "839033", "10.0.1.4:11211" },
      { "839266", "10.0.1.1:11211" }, { "839499", "10.0.1.5:11211" },
      { "839732", "10.0.1.3:11211" }, { "839965", "10.0.1.7:11211" },
      { "840198", "10.0.1.7:11211" }, { "840431", "10.0.1.7:11211" },
      { "840664", "10.0.1.6:11211" }, { "840897", "10.0.1.1:11211" },
      { "841130", "10.0.1.7:11211" }, { "841363", "10.0.1.2:11211" },
      { "841596", "10.0.1.6:11211" }, { "841829", "10.0.1.7:11211" },
      { "842062", "10.0.1.4:11211" }, { "842295", "10.0.1.7:11211" },
      { "842528", "10.0.1.7:11211" }, { "842761", "10.0.1.2:11211" },
      { "842994", "10.0.1.7:11211" }, { "843227", "10.0.1.7:11211" },
      { "843460", "10.0.1.5:11211" }, { "843693", "10.0.1.1:11211" },
      { "843926", "10.0.1.5:11211" }, { "844159", "10.0.1.1:11211" },
      { "844392", "10.0.1.6:11211" }, { "844625", "10.0.1.1:11211" },
      { "844858", "10.0.1.5:11211" }, { "845091", "10.0.1.8:11211" },
      { "845324", "10.0.1.6:11211" }, { "845557", "10.0.1.8:11211" },
      { "845790", "10.0.1.4:11211" }, { "846023", "10.0.1.1:11211" },
      { "846256", "10.0.1.2:11211" }, { "846489", "10.0.1.8:11211" },
      { "846722", "10.0.1.4:11211" }, { "846955", "10.0.1.2:11211" },
      { "847188", "10.0.1.6:11211" }, { "847421", "10.0.1.1:11211" },
      { "847654", "10.0.1.2:11211" }, { "847887", "10.0.1.1:11211" },
      { "848120", "10.0.1.8:11211" }, { "848353", "10.0.1.6:11211" },
      { "848586", "10.0.1.6:11211" }, { "848819", "10.0.1.2:11211" },
      { "849052", "10.0.1.3:11211" }, { "849285", "10.0.1.1:11211" },
      { "849518", "10.0.1.5:11211" }, { "849751", "10.0.1.3:11211" },
      { "849984", "10.0.1.6:11211" }, { "850217", "10.0.1.2:11211" },
      { "855576", "10.0.1.8:11211" }, { "855809", "10.0.1.2:11211" },
      { "856042", "10.0.1.7:11211" }, { "856275", "10.0.1.6:11211" },
      { "856508", "10.0.1.7:11211" }, { "856741", "10.0.1.3:11211" },
      { "856974", "10.0.1.2:11211" }, { "857207", "10.0.1.4:11211" },
      { "857440", "10.0.1.7:11211" }, { "857673", "10.0.1.5:11211" },
      { "857906", "10.0.1.8:11211" }, { "858139", "10.0.1.3:11211" },
      { "858372", "10.0.1.1:11211" }, { "858605", "10.0.1.1:11211" },
      { "858838", "10.0.1.6:11211" }, { "859071", "10.0.1.5:11211" },
      { "859304", "10.0.1.3:11211" }, { "859537", "10.0.1.2:11211" },
      { "859770", "10.0.1.8:11211" }, { "860003", "10.0.1.8:11211" },
      { "860236", "10.0.1.4:11211" }, { "860469", "10.0.1.2:11211" },
      { "860702", "10.0.1.7:11211" }, { "860935", "10.0.1.6:11211" },
      { "861168", "10.0.1.4:11211" }, { "861401", "10.0.1.2:11211" },
      { "861634", "10.0.1.5:11211" }, { "861867", "10.0.1.5:11211" },
      { "862100", "10.0.1.1:11211" }, { "862333", "10.0.1.8:11211" },
      { "862566", "10.0.1.4:11211" }, { "862799", "10.0.1.1:11211" },
      { "863032", "10.0.1.4:11211" }, { "863265", "10.0.1.3:11211" },
      { "863498", "10.0.1.6:11211" }, { "863731", "10.0.1.8:11211" },
      { "863964", "10.0.1.6:11211" }, { "864197", "10.0.1.5:11211" },
      { "864430", "10.0.1.6:11211" }, { "864663", "10.0.1.1:11211" },
      { "864896", "10.0.1.6:11211" }, { "865129", "10.0.1.3:11211" },
      { "865362", "10.0.1.8:11211" }, { "865595", "10.0.1.4:11211" },
      { "865828", "10.0.1.8:11211" }, { "866061", "10.0.1.5:11211" },
      { "866294", "10.0.1.3:11211" }, { "866527", "10.0.1.5:11211" },
      { "866760", "10.0.1.3:11211" }, { "866993", "10.0.1.2:11211" },
      { "867226", "10.0.1.1:11211" }, { "867459", "10.0.1.1:11211" },
      { "867692", "10.0.1.6:11211" }, { "867925", "10.0.1.6:11211" },
      { "868158", "10.0.1.4:11211" }, { "868391", "10.0.1.8:11211" },
      { "868624", "10.0.1.3:11211" }, { "868857", "10.0.1.5:11211" },
      { "869090", "10.0.1.4:11211" }, { "869323", "10.0.1.7:11211" },
      { "869556", "10.0.1.1:11211" }, { "869789", "10.0.1.2:11211" },
      { "870022", "10.0.1.1:11211" }, { "870255", "10.0.1.4:11211" },
      { "870488", "10.0.1.7:11211" }, { "870721", "10.0.1.1:11211" },
      { "870954", "10.0.1.7:11211" }, { "871187", "10.0.1.1:11211" },
      { "871420", "10.0.1.4:11211" }, { "871653", "10.0.1.4:11211" },
      { "871886", "10.0.1.8:11211" }, { "872119", "10.0.1.6:11211" },
      { "872352", "10.0.1.6:11211" }, { "872585", "10.0.1.1:11211" },
      { "872818", "10.0.1.3:11211" }, { "873051", "10.0.1.4:11211" },
      { "873284", "10.0.1.8:11211" }, { "873517", "10.0.1.3:11211" },
      { "873750", "10.0.1.8:11211" }, { "877711", "10.0.1.6:11211" },
      { "877944", "10.0.1.6:11211" }, { "878177", "10.0.1.6:11211" },
      { "878410", "10.0.1.3:11211" }, { "878643", "10.0.1.6:11211" },
      { "878876", "10.0.1.5:11211" }, { "879109", "10.0.1.4:11211" },
      { "879342", "10.0.1.7:11211" }, { "879575", "10.0.1.6:11211" },
      { "879808", "10.0.1.1:11211" }, { "880041", "10.0.1.7:11211" },
      { "880274", "10.0.1.8:11211" }, { "880507", "10.0.1.2:11211" },
      { "880740", "10.0.1.7:11211" }, { "880973", "10.0.1.5:11211" },
      { "881206", "10.0.1.8:11211" }, { "881439", "10.0.1.6:11211" },
      { "881672", "10.0.1.6:11211" }, { "881905", "10.0.1.2:11211" },
      { "882138", "10.0.1.8:11211" }, { "882371", "10.0.1.3:11211" },
      { "882604", "10.0.1.1:11211" }, { "882837", "10.0.1.2:11211" },
      { "883070", "10.0.1.8:11211" }, { "883303", "10.0.1.3:11211" },
      { "883536", "10.0.1.1:11211" }, { "883769", "10.0.1.5:11211" },
      { "884002", "10.0.1.2:11211" }, { "884235", "10.0.1.5:11211" },
      { "884468", "10.0.1.4:11211" }, { "884701", "10.0.1.3:11211" },
      { "884934", "10.0.1.7:11211" }, { "885167", "10.0.1.4:11211" },
      { "885400", "10.0.1.7:11211" }, { "885633", "10.0.1.8:11211" },
      { "885866", "10.0.1.7:11211" }, { "886099", "10.0.1.4:11211" },
      { "886332", "10.0.1.4:11211" }, { "886565", "10.0.1.7:11211" },
      { "886798", "10.0.1.3:11211" }, { "887031", "10.0.1.3:11211" },
      { "887264", "10.0.1.6:11211" }, { "887497", "10.0.1.7:11211" },
      { "887730", "10.0.1.7:11211" }, { "887963", "10.0.1.7:11211" },
      { "888196", "10.0.1.3:11211" }, { "888429", "10.0.1.8:11211" },
      { "888662", "10.0.1.5:11211" }, { "888895", "10.0.1.4:11211" },
      { "889128", "10.0.1.3:11211" }, { "889361", "10.0.1.3:11211" },
      { "889594", "10.0.1.4:11211" }, { "889827", "10.0.1.5:11211" },
      { "890060", "10.0.1.7:11211" }, { "890293", "10.0.1.3:11211" },
      { "890526", "10.0.1.1:11211" }, { "890759", "10.0.1.3:11211" },
      { "890992", "10.0.1.3:11211" }, { "891225", "10.0.1.1:11211" },
      { "891458", "10.0.1.6:11211" }, { "891691", "10.0.1.6:11211" },
      { "891924", "10.0.1.2:11211" }, { "892157", "10.0.1.4:11211" },
      { "892390", "10.0.1.4:11211" }, { "892623", "10.0.1.8:11211" },
      { "892856", "10.0.1.1:11211" }, { "893089", "10.0.1.7:11211" },
      { "893322", "10.0.1.8:11211" }, { "893555", "10.0.1.5:11211" },
      { "893788", "10.0.1.6:11211" }, { "894021", "10.0.1.8:11211" },
      { "894254", "10.0.1.6:11211" }, { "894487", "10.0.1.4:11211" },
      { "894720", "10.0.1.6:11211" }, { "894953", "10.0.1.6:11211" },
      { "895186", "10.0.1.3:11211" }, { "895419", "10.0.1.1:11211" },
      { "895652", "10.0.1.8:11211" }, { "895885", "10.0.1.2:11211" },
      { "896118", "10.0.1.7:11211" }, { "896351", "10.0.1.3:11211" },
      { "896584", "10.0.1.2:11211" }, { "896817", "10.0.1.1:11211" },
      { "897050", "10.0.1.2:11211" }, { "897283", "10.0.1.2:11211" },
      { "897516", "10.0.1.4:11211" }, { "897749", "10.0.1.5:11211" },
      { "897982", "10.0.1.6:11211" }, { "898215", "10.0.1.5:11211" },
      { "898448", "10.0.1.7:11211" }, { "898681", "10.0.1.2:11211" },
      { "898914", "10.0.1.2:11211" }, { "899147", "10.0.1.4:11211" },
      { "899380", "10.0.1.5:11211" }, { "899613", "10.0.1.1:11211" },
      { "899846", "10.0.1.2:11211" }, { "900079", "10.0.1.3:11211" },
      { "900312", "10.0.1.1:11211" }, { "900545", "10.0.1.6:11211" },
      { "900778", "10.0.1.6:11211" }, { "901011", "10.0.1.2:11211" },
      { "901244", "10.0.1.7:11211" }, { "901477", "10.0.1.6:11211" },
      { "901710", "10.0.1.2:11211" }, { "901943", "10.0.1.8:11211" },
      { "902176", "10.0.1.6:11211" }, { "902409", "10.0.1.7:11211" },
      { "902642", "10.0.1.4:11211" }, { "902875", "10.0.1.5:11211" },
      { "903108", "10.0.1.6:11211" }, { "907535", "10.0.1.1:11211" },
      { "907768", "10.0.1.3:11211" }, { "908001", "10.0.1.6:11211" },
      { "908234", "10.0.1.5:11211" }, { "908467", "10.0.1.2:11211" },
      { "908700", "10.0.1.8:11211" }, { "908933", "10.0.1.8:11211" },
      { "909166", "10.0.1.2:11211" }, { "909399", "10.0.1.2:11211" },
      { "909632", "10.0.1.7:11211" }, { "909865", "10.0.1.3:11211" },
      { "910098", "10.0.1.2:11211" }, { "910331", "10.0.1.6:11211" },
      { "910564", "10.0.1.2:11211" }, { "910797", "10.0.1.5:11211" },
      { "911030", "10.0.1.8:11211" }, { "911263", "10.0.1.7:11211" },
      { "911496", "10.0.1.2:11211" }, { "911729", "10.0.1.2:11211" },
      { "911962", "10.0.1.1:11211" }, { "912195", "10.0.1.5:11211" },
      { "912428", "10.0.1.8:11211" }, { "912661", "10.0.1.8:11211" },
      { "912894", "10.0.1.1:11211" }, { "913127", "10.0.1.8:11211" },
      { "913360", "10.0.1.7:11211" }, { "913593", "10.0.1.8:11211" },
      { "913826", "10.0.1.1:11211" }, { "914059", "10.0.1.2:11211" },
      { "914292", "10.0.1.8:11211" }, { "914525", "10.0.1.5:11211" },
      { "914758", "10.0.1.1:11211" }, { "914991", "10.0.1.4:11211" },
      { "915224", "10.0.1.7:11211" }, { "915457", "10.0.1.1:11211" },
      { "915690", "10.0.1.2:11211" }, { "915923", "10.0.1.1:11211" },
      { "916156", "10.0.1.8:11211" }, { "916389", "10.0.1.6:11211" },
      { "916622", "10.0.1.8:11211" }, { "916855", "10.0.1.5:11211" },
      { "917088", "10.0.1.6:11211" }, { "917321", "10.0.1.2:11211" },
      { "917554", "10.0.1.8:11211" }, { "917787", "10.0.1.3:11211" },
      { "918020", "10.0.1.3:11211" }, { "918253", "10.0.1.1:11211" },
      { "918486", "10.0.1.1:11211" }, { "918719", "10.0.1.5:11211" },
      { "918952", "10.0.1.1:11211" }, { "919185", "10.0.1.2:11211" },
      { "919418", "10.0.1.3:11211" }, { "919651", "10.0.1.6:11211" },
      { "919884", "10.0.1.7:11211" }, { "920117", "10.0.1.8:11211" },
      { "920350", "10.0.1.4:11211" }, { "920583", "10.0.1.7:11211" },
      { "920816", "10.0.1.2:11211" }, { "921049", "10.0.1.6:11211" },
      { "921282", "10.0.1.6:11211" }, { "921515", "10.0.1.2:11211" },
      { "921748", "10.0.1.2:11211" }, { "921981", "10.0.1.1:11211" },
      { "922214", "10.0.1.7:11211" }, { "922447", "10.0.1.5:11211" },
      { "922680", "10.0.1.7:11211" }, { "922913", "10.0.1.6:11211" },
      { "923146", "10.0.1.3:11211" }, { "923379", "10.0.1.1:11211" },
      { "923612", "10.0.1.5:11211" }, { "923845", "10.0.1.7:11211" },
      { "924078", "10.0.1.4:11211" }, { "924311", "10.0.1.7:11211" },
      { "924544", "10.0.1.2:11211" }, { "924777", "10.0.1.7:11211" },
      { "925010", "10.0.1.8:11211" }, { "925243", "10.0.1.8:11211" },
      { "925476", "10.0.1.7:11211" }, { "925709", "10.0.1.8:11211" },
      { "925942", "10.0.1.6:11211" }, { "926175", "10.0.1.4:11211" },
      { "926408", "10.0.1.3:11211" }, { "926641", "10.0.1.2:11211" },
      { "926874", "10.0.1.5:11211" }, { "927107", "10.0.1.3:11211" },
      { "927340", "10.0.1.1:11211" }, { "927573", "10.0.1.3:11211" },
      { "927806", "10.0.1.3:11211" }, { "932699", "10.0.1.4:11211" },
      { "932932", "10.0.1.7:11211" }, { "933165", "10.0.1.3:11211" },
      { "933398", "10.0.1.8:11211" }, { "933631", "10.0.1.5:11211" },
      { "933864", "10.0.1.6:11211" }, { "934097", "10.0.1.8:11211" },
      { "934330", "10.0.1.5:11211" }, { "934563", "10.0.1.1:11211" },
      { "934796", "10.0.1.4:11211" }, { "935029", "10.0.1.2:11211" },
      { "935262", "10.0.1.6:11211" }, { "935495", "10.0.1.3:11211" },
      { "935728", "10.0.1.6:11211" }, { "935961", "10.0.1.3:11211" },
      { "936194", "10.0.1.6:11211" }, { "936427", "10.0.1.6:11211" },
      { "936660", "10.0.1.5:11211" }, { "936893", "10.0.1.7:11211" },
      { "937126", "10.0.1.8:11211" }, { "937359", "10.0.1.6:11211" },
      { "937592", "10.0.1.7:11211" }, { "937825", "10.0.1.3:11211" },
      { "938058", "10.0.1.7:11211" }, { "938291", "10.0.1.7:11211" },
      { "938524", "10.0.1.2:11211" }, { "938757", "10.0.1.6:11211" },
      { "938990", "10.0.1.2:11211" }, { "939223", "10.0.1.7:11211" },
      { "939456", "10.0.1.4:11211" }, { "939689", "10.0.1.1:11211" },
      { "939922", "10.0.1.5:11211" }, { "940155", "10.0.1.2:11211" },
      { "940388", "10.0.1.1:11211" }, { "940621", "10.0.1.4:11211" },
      { "940854", "10.0.1.3:11211" }, { "941087", "10.0.1.4:11211" },
      { "944815", "10.0.1.4:11211" }, { "945048", "10.0.1.1:11211" },
      { "945281", "10.0.1.2:11211" }, { "945514", "10.0.1.4:11211" },
      { "945747", "10.0.1.5:11211" }, { "945980", "10.0.1.4:11211" },
      { "946213", "10.0.1.2:11211" }, { "946446", "10.0.1.2:11211" },
      { "946679", "10.0.1.1:11211" }, { "946912", "10.0.1.3:11211" },
      { "947145", "10.0.1.3:11211" }, { "947378", "10.0.1.1:11211" },
      { "947611", "10.0.1.3:11211" }, { "947844", "10.0.1.2:11211" },
      { "948077", "10.0.1.3:11211" }, { "948310", "10.0.1.7:11211" },
      { "948543", "10.0.1.5:11211" }, { "948776", "10.0.1.7:11211" },
      { "949009", "10.0.1.6:11211" }, { "949242", "10.0.1.1:11211" },
      { "949475", "10.0.1.4:11211" }, { "949708", "10.0.1.2:11211" },
      { "949941", "10.0.1.5:11211" }, { "950174", "10.0.1.1:11211" },
      { "950407", "10.0.1.4:11211" }, { "950640", "10.0.1.7:11211" },
      { "950873", "10.0.1.3:11211" }, { "951106", "10.0.1.3:11211" },
      { "951339", "10.0.1.1:11211" }, { "951572", "10.0.1.7:11211" },
      { "951805", "10.0.1.7:11211" }, { "952038", "10.0.1.1:11211" },
      { "952271", "10.0.1.8:11211" }, { "952504", "10.0.1.8:11211" },
      { "952737", "10.0.1.7:11211" }, { "952970", "10.0.1.1:11211" },
      { "953203", "10.0.1.2:11211" }, { "953436", "10.0.1.5:11211" },
      { "953669", "10.0.1.2:11211" }, { "953902", "10.0.1.4:11211" },
      { "954135", "10.0.1.1:11211" }, { "954368", "10.0.1.6:11211" },
      { "954601", "10.0.1.7:11211" }, { "954834", "10.0.1.3:11211" },
      { "955067", "10.0.1.6:11211" }, { "955300", "10.0.1.7:11211" },
      { "955533", "10.0.1.3:11211" }, { "955766", "10.0.1.1:11211" },
      { "955999", "10.0.1.8:11211" }, { "956232", "10.0.1.3:11211" },
      { "956465", "10.0.1.4:11211" }, { "956698", "10.0.1.5:11211" },
      { "956931", "10.0.1.2:11211" }, { "957164", "10.0.1.5:11211" },
      { "957397", "10.0.1.6:11211" }, { "957630", "10.0.1.7:11211" },
      { "957863", "10.0.1.7:11211" }, { "958096", "10.0.1.8:11211" },
      { "958329", "10.0.1.4:11211" }, { "958562", "10.0.1.8:11211" },
      { "958795", "10.0.1.4:11211" }, { "959028", "10.0.1.6:11211" },
      { "959261", "10.0.1.5:11211" }, { "959494", "10.0.1.1:11211" },
      { "959727", "10.0.1.6:11211" }, { "959960", "10.0.1.5:11211" },
      { "960193", "10.0.1.4:11211" }, { "960426", "10.0.1.8:11211" },
      { "960659", "10.0.1.2:11211" }, { "960892", "10.0.1.8:11211" },
      { "961125", "10.0.1.8:11211" }, { "961358", "10.0.1.2:11211" },
      { "961591", "10.0.1.2:11211" }, { "961824", "10.0.1.5:11211" },
      { "962057", "10.0.1.1:11211" }, { "962290", "10.0.1.3:11211" },
      { "962523", "10.0.1.6:11211" }, { "962756", "10.0.1.6:11211" },
      { "962989", "10.0.1.6:11211" }, { "963222", "10.0.1.4:11211" },
      { "963455", "10.0.1.2:11211" }, { "963688", "10.0.1.1:11211" },
      { "963921", "10.0.1.6:11211" }, { "964154", "10.0.1.3:11211" },
      { "964387", "10.0.1.1:11211" }, { "964620", "10.0.1.7:11211" },
      { "964853", "10.0.1.2:11211" }, { "965086", "10.0.1.5:11211" },
      { "965319", "10.0.1.5:11211" }, { "965552", "10.0.1.7:11211" },
      { "965785", "10.0.1.7:11211" }, { "966018", "10.0.1.3:11211" },
      { "966251", "10.0.1.6:11211" }, { "966484", "10.0.1.8:11211" },
      { "966717", "10.0.1.7:11211" }, { "966950", "10.0.1.6:11211" },
      { "967183", "10.0.1.3:11211" }, { "967416", "10.0.1.1:11211" },
      { "967649", "10.0.1.2:11211" }, { "967882", "10.0.1.8:11211" },
      { "968115", "10.0.1.7:11211" }, { "968348", "10.0.1.3:11211" },
      { "968581", "10.0.1.4:11211" }, { "968814", "10.0.1.4:11211" },
      { "969047", "10.0.1.3:11211" }, { "969280", "10.0.1.7:11211" },
      { "969513", "10.0.1.6:11211" }, { "969746", "10.0.1.1:11211" },
      { "969979", "10.0.1.4:11211" }, { "970212", "10.0.1.1:11211" },
      { "970445", "10.0.1.6:11211" }, { "970678", "10.0.1.1:11211" },
      { "970911", "10.0.1.3:11211" }, { "971144", "10.0.1.6:11211" },
      { "971377", "10.0.1.1:11211" }, { "971610", "10.0.1.1:11211" },
      { "971843", "10.0.1.4:11211" }, { "972076", "10.0.1.4:11211" },
      { "972309", "10.0.1.3:11211" }, { "976037", "10.0.1.1:11211" },
      { "976270", "10.0.1.2:11211" }, { "976503", "10.0.1.6:11211" },
      { "976736", "10.0.1.7:11211" }, { "976969", "10.0.1.7:11211" },
      { "977202", "10.0.1.7:11211" }, { "977435", "10.0.1.6:11211" },
      { "977668", "10.0.1.6:11211" }, { "977901", "10.0.1.2:11211" },
      { "978134", "10.0.1.5:11211" }, { "978367", "10.0.1.6:11211" },
      { "978600", "10.0.1.5:11211" }, { "978833", "10.0.1.1:11211" },
      { "979066", "10.0.1.5:11211" }, { "979299", "10.0.1.5:11211" },
      { "979532", "10.0.1.3:11211" }, { "979765", "10.0.1.4:11211" },
      { "979998", "10.0.1.8:11211" }, { "980231", "10.0.1.3:11211" },
      { "980464", "10.0.1.8:11211" }, { "980697", "10.0.1.1:11211" },
      { "980930", "10.0.1.2:11211" }, { "981163", "10.0.1.5:11211" },
      { "987454", "10.0.1.2:11211" }, { "987687", "10.0.1.2:11211" },
      { "987920", "10.0.1.7:11211" }, { "988153", "10.0.1.1:11211" },
      { "988386", "10.0.1.1:11211" }, { "988619", "10.0.1.6:11211" },
      { "988852", "10.0.1.3:11211" }, { "989085", "10.0.1.5:11211" },
      { "989318", "10.0.1.7:11211" }, { "989551", "10.0.1.5:11211" },
      { "989784", "10.0.1.8:11211" }, { "990017", "10.0.1.1:11211" },
      { "990250", "10.0.1.5:11211" }, { "990483", "10.0.1.8:11211" },
      { "990716", "10.0.1.5:11211" }, { "990949", "10.0.1.4:11211" },
      { "991182", "10.0.1.8:11211" }, { "991415", "10.0.1.3:11211" },
      { "991648", "10.0.1.4:11211" }, { "991881", "10.0.1.5:11211" },
      { "992114", "10.0.1.5:11211" }, { "992347", "10.0.1.3:11211" },
      { "992580", "10.0.1.7:11211" }, { "992813", "10.0.1.4:11211" },
      { "993046", "10.0.1.6:11211" }, { "993279", "10.0.1.3:11211" },
      { "993512", "10.0.1.5:11211" }, { "993745", "10.0.1.4:11211" },
      { "993978", "10.0.1.7:11211" }, { "994211", "10.0.1.7:11211" },
      { "994444", "10.0.1.5:11211" }, { "994677", "10.0.1.1:11211" },
      { "994910", "10.0.1.7:11211" }, { "995143", "10.0.1.7:11211" },
      { "995376", "10.0.1.4:11211" }, { "995609", "10.0.1.1:11211" },
      { "995842", "10.0.1.6:11211" }, { "996075", "10.0.1.6:11211" },
      { "996308", "10.0.1.6:11211" }, { "996541", "10.0.1.2:11211" },
      { "996774", "10.0.1.6:11211" }, { "997007", "10.0.1.7:11211" },
      { "997240", "10.0.1.2:11211" }, { "997473", "10.0.1.1:11211" },
      { "997706", "10.0.1.4:11211" }, { "999104", "10.0.1.8:11211" },
      { "999337", "10.0.1.4:11211" }, { "999570", "10.0.1.6:11211" },
      { "999803", "10.0.1.4:11211" } };

    for (String[] s : exp) {
      String k = s[0];
      String server = s[1];
      MemcachedNode n = locator.getPrimary(k);
      assertEquals("/" + server, n.getSocketAddress().toString());
    }

  }

  public void testLibKetamaCompatThree() {
	    String[] servers = {"10.0.1.1:11211", "10.0.1.2:11211",
	      "10.0.1.3:11211", "10.0.1.4:11211", "10.0.1.5:11211",
	      "10.0.1.6:11211", "10.0.1.7:11211", "10.0.1.8:11211"};
	    locator = new KetamaNodeLocator(Arrays.asList(mockNodes(servers)),
	        DefaultHashAlgorithm.KETAMA_HASH);

	    String[][] exp = { { "0", "10.0.1.1:11211" }, { "233", "10.0.1.7:11211" },
	      { "466", "10.0.1.3:11211" }, { "699", "10.0.1.1:11211" },
	      { "932", "10.0.1.6:11211" }, { "1165", "10.0.1.2:11211" },
	      { "1398", "10.0.1.1:11211" }, { "1631", "10.0.1.6:11211" },
	      { "1864", "10.0.1.5:11211" }, { "2097", "10.0.1.3:11211" },
	      { "2330", "10.0.1.7:11211" }, { "2563", "10.0.1.3:11211" },
	      { "2796", "10.0.1.6:11211" }, { "3029", "10.0.1.1:11211" },
	      { "3262", "10.0.1.2:11211" }, { "3495", "10.0.1.3:11211" },
	      { "3728", "10.0.1.8:11211" }, { "3961", "10.0.1.4:11211" },
	      { "4194", "10.0.1.4:11211" }, { "4427", "10.0.1.3:11211" },
	      { "4660", "10.0.1.4:11211" }, { "4893", "10.0.1.7:11211" },
	      { "5126", "10.0.1.4:11211" }, { "5359", "10.0.1.2:11211" },
	      { "5592", "10.0.1.2:11211" }, { "5825", "10.0.1.3:11211" },
	      { "6058", "10.0.1.2:11211" }, { "6291", "10.0.1.7:11211" },
	      { "6524", "10.0.1.5:11211" }, { "6757", "10.0.1.5:11211" },
	      { "6990", "10.0.1.1:11211" }, { "7223", "10.0.1.5:11211" },
	      { "7456", "10.0.1.4:11211" }, { "7689", "10.0.1.2:11211" },
	      { "7922", "10.0.1.5:11211" }, { "8155", "10.0.1.5:11211" },
	      { "8388", "10.0.1.1:11211" }, { "8621", "10.0.1.2:11211" },
	      { "8854", "10.0.1.2:11211" }, { "9087", "10.0.1.1:11211" },
	      { "9320", "10.0.1.6:11211" }, { "9553", "10.0.1.3:11211" },
	      { "9786", "10.0.1.2:11211" }, { "10019", "10.0.1.5:11211" },
	      { "10252", "10.0.1.1:11211" }, { "10485", "10.0.1.5:11211" },
	      { "10718", "10.0.1.5:11211" }, { "10951", "10.0.1.2:11211" },
	      { "11184", "10.0.1.5:11211" }, { "11417", "10.0.1.3:11211" },
	      { "11650", "10.0.1.8:11211" }, { "11883", "10.0.1.2:11211" },
	      { "12116", "10.0.1.2:11211" }, { "12349", "10.0.1.7:11211" },
	      { "12582", "10.0.1.5:11211" }, { "12815", "10.0.1.3:11211" },
	      { "13048", "10.0.1.8:11211" }, { "13281", "10.0.1.6:11211" },
	      { "13514", "10.0.1.3:11211" }, { "13747", "10.0.1.6:11211" },
	      { "13980", "10.0.1.6:11211" }, { "14213", "10.0.1.8:11211" },
	      { "14446", "10.0.1.2:11211" }, { "14679", "10.0.1.3:11211" },
	      { "14912", "10.0.1.7:11211" }, { "15145", "10.0.1.1:11211" },
	      { "15378", "10.0.1.4:11211" }, { "15611", "10.0.1.1:11211" },
	      { "15844", "10.0.1.3:11211" }, { "16077", "10.0.1.3:11211" },
	      { "16310", "10.0.1.5:11211" }, { "16543", "10.0.1.5:11211" },
	      { "16776", "10.0.1.4:11211" }, { "17009", "10.0.1.1:11211" },
	      { "17242", "10.0.1.4:11211" }, { "17475", "10.0.1.7:11211" },
	      { "17708", "10.0.1.6:11211" }, { "17941", "10.0.1.2:11211" },
	      { "28892", "10.0.1.7:11211" }, { "29125", "10.0.1.4:11211" },
	      { "29358", "10.0.1.7:11211" }, { "29591", "10.0.1.5:11211" },
	      { "29824", "10.0.1.7:11211" }, { "30057", "10.0.1.7:11211" },
	      { "30290", "10.0.1.3:11211" }, { "30523", "10.0.1.8:11211" },
	      { "30756", "10.0.1.3:11211" }, { "30989", "10.0.1.4:11211" },
	      { "31222", "10.0.1.6:11211" }, { "31455", "10.0.1.1:11211" },
	      { "31688", "10.0.1.2:11211" }, { "31921", "10.0.1.2:11211" },
	      { "32154", "10.0.1.8:11211" }, { "32387", "10.0.1.1:11211" },
	      { "32620", "10.0.1.3:11211" }, { "32853", "10.0.1.2:11211" },
	      { "33086", "10.0.1.7:11211" }, { "33319", "10.0.1.4:11211" },
	      { "33552", "10.0.1.5:11211" }, { "33785", "10.0.1.3:11211" },
	      { "34018", "10.0.1.3:11211" }, { "34251", "10.0.1.6:11211" },
	      { "34484", "10.0.1.7:11211" }, { "34717", "10.0.1.1:11211" },
	      { "34950", "10.0.1.3:11211" }, { "35183", "10.0.1.6:11211" },
	      { "35416", "10.0.1.8:11211" }, { "35649", "10.0.1.5:11211" },
	      { "35882", "10.0.1.7:11211" }, { "36115", "10.0.1.2:11211" },
	      { "36348", "10.0.1.5:11211" }, { "36581", "10.0.1.7:11211" },
	      { "36814", "10.0.1.8:11211" }, { "37047", "10.0.1.8:11211" },
	      { "37280", "10.0.1.8:11211" }, { "42173", "10.0.1.8:11211" },
	      { "42406", "10.0.1.3:11211" }, { "47998", "10.0.1.2:11211" },
	      { "48231", "10.0.1.5:11211" }, { "48464", "10.0.1.5:11211" },
	      { "48697", "10.0.1.3:11211" }, { "48930", "10.0.1.1:11211" },
	      { "49163", "10.0.1.2:11211" }, { "49396", "10.0.1.8:11211" },
	      { "49629", "10.0.1.1:11211" }, { "49862", "10.0.1.8:11211" },
	      { "50095", "10.0.1.5:11211" }, { "50328", "10.0.1.2:11211" },
	      { "50561", "10.0.1.5:11211" }, { "50794", "10.0.1.7:11211" },
	      { "51027", "10.0.1.3:11211" }, { "51260", "10.0.1.5:11211" },
	      { "51493", "10.0.1.3:11211" }, { "51726", "10.0.1.8:11211" },
	      { "51959", "10.0.1.2:11211" }, { "52192", "10.0.1.8:11211" },
	      { "56153", "10.0.1.2:11211" }, { "56386", "10.0.1.6:11211" },
	      { "56619", "10.0.1.8:11211" }, { "56852", "10.0.1.6:11211" },
	      { "57085", "10.0.1.2:11211" }, { "57318", "10.0.1.7:11211" },
	      { "57551", "10.0.1.8:11211" }, { "57784", "10.0.1.4:11211" },
	      { "58017", "10.0.1.6:11211" }, { "58250", "10.0.1.8:11211" },
	      { "58483", "10.0.1.8:11211" }, { "58716", "10.0.1.6:11211" },
	      { "58949", "10.0.1.7:11211" }, { "59182", "10.0.1.3:11211" },
	      { "59415", "10.0.1.2:11211" }, { "59648", "10.0.1.7:11211" },
	      { "59881", "10.0.1.8:11211" }, { "60114", "10.0.1.8:11211" },
	      { "60347", "10.0.1.3:11211" }, { "60580", "10.0.1.6:11211" },
	      { "60813", "10.0.1.8:11211" }, { "61046", "10.0.1.6:11211" },
	      { "61279", "10.0.1.7:11211" }, { "61512", "10.0.1.5:11211" },
	      { "61745", "10.0.1.7:11211" }, { "61978", "10.0.1.8:11211" },
	      { "62211", "10.0.1.7:11211" }, { "62444", "10.0.1.1:11211" },
	      { "62677", "10.0.1.7:11211" }, { "62910", "10.0.1.3:11211" },
	      { "63143", "10.0.1.2:11211" }, { "63376", "10.0.1.2:11211" },
	      { "63609", "10.0.1.6:11211" }, { "63842", "10.0.1.2:11211" },
	      { "64075", "10.0.1.5:11211" }, { "64308", "10.0.1.6:11211" },
	      { "64541", "10.0.1.5:11211" }, { "64774", "10.0.1.4:11211" },
	      { "65007", "10.0.1.7:11211" }, { "65240", "10.0.1.7:11211" },
	      { "65473", "10.0.1.6:11211" }, { "65706", "10.0.1.8:11211" },
	      { "65939", "10.0.1.4:11211" }, { "66172", "10.0.1.1:11211" },
	      { "66405", "10.0.1.2:11211" }, { "66638", "10.0.1.6:11211" },
	      { "66871", "10.0.1.5:11211" }, { "67104", "10.0.1.2:11211" },
	      { "67337", "10.0.1.8:11211" }, { "67570", "10.0.1.8:11211" },
	      { "67803", "10.0.1.5:11211" }, { "68036", "10.0.1.8:11211" },
	      { "68269", "10.0.1.4:11211" }, { "68502", "10.0.1.7:11211" },
	      { "68735", "10.0.1.1:11211" }, { "68968", "10.0.1.6:11211" },
	      { "69201", "10.0.1.6:11211" }, { "69434", "10.0.1.6:11211" },
	      { "69667", "10.0.1.3:11211" }, { "69900", "10.0.1.2:11211" },
	      { "70133", "10.0.1.8:11211" }, { "70366", "10.0.1.2:11211" },
	      { "70599", "10.0.1.2:11211" }, { "70832", "10.0.1.1:11211" },
	      { "71065", "10.0.1.5:11211" }, { "71298", "10.0.1.2:11211" },
	      { "71531", "10.0.1.2:11211" }, { "71764", "10.0.1.5:11211" },
	      { "71997", "10.0.1.5:11211" }, { "72230", "10.0.1.2:11211" },
	      { "72463", "10.0.1.7:11211" }, { "72696", "10.0.1.6:11211" },
	      { "72929", "10.0.1.4:11211" }, { "73162", "10.0.1.4:11211" },
	      { "73395", "10.0.1.7:11211" }, { "73628", "10.0.1.7:11211" },
	      { "73861", "10.0.1.1:11211" }, { "74094", "10.0.1.6:11211" },
	      { "74327", "10.0.1.1:11211" }, { "74560", "10.0.1.6:11211" },
	      { "74793", "10.0.1.1:11211" }, { "75026", "10.0.1.5:11211" },
	      { "75259", "10.0.1.5:11211" }, { "75492", "10.0.1.8:11211" },
	      { "75725", "10.0.1.7:11211" }, { "75958", "10.0.1.4:11211" },
	      { "76191", "10.0.1.5:11211" }, { "76424", "10.0.1.5:11211" },
	      { "76657", "10.0.1.2:11211" }, { "76890", "10.0.1.7:11211" },
	      { "77123", "10.0.1.4:11211" }, { "77356", "10.0.1.2:11211" },
	      { "77589", "10.0.1.6:11211" }, { "77822", "10.0.1.1:11211" },
	      { "78055", "10.0.1.6:11211" }, { "78288", "10.0.1.7:11211" },
	      { "78521", "10.0.1.7:11211" }, { "78754", "10.0.1.5:11211" },
	      { "78987", "10.0.1.6:11211" }, { "79220", "10.0.1.4:11211" },
	      { "79453", "10.0.1.6:11211" }, { "79686", "10.0.1.4:11211" },
	      { "79919", "10.0.1.3:11211" }, { "80152", "10.0.1.2:11211" },
	      { "80385", "10.0.1.6:11211" }, { "80618", "10.0.1.5:11211" },
	      { "80851", "10.0.1.7:11211" }, { "81084", "10.0.1.8:11211" },
	      { "81317", "10.0.1.5:11211" }, { "81550", "10.0.1.8:11211" },
	      { "81783", "10.0.1.4:11211" }, { "82016", "10.0.1.8:11211" },
	      { "82249", "10.0.1.5:11211" }, { "82482", "10.0.1.5:11211" },
	      { "82715", "10.0.1.5:11211" }, { "82948", "10.0.1.5:11211" },
	      { "83181", "10.0.1.1:11211" }, { "83414", "10.0.1.1:11211" },
	      { "83647", "10.0.1.2:11211" }, { "83880", "10.0.1.2:11211" },
	      { "84113", "10.0.1.6:11211" }, { "84346", "10.0.1.6:11211" },
	      { "84579", "10.0.1.5:11211" }, { "84812", "10.0.1.8:11211" },
	      { "85045", "10.0.1.6:11211" }, { "85278", "10.0.1.7:11211" },
	      { "85511", "10.0.1.2:11211" }, { "85744", "10.0.1.1:11211" },
	      { "85977", "10.0.1.6:11211" }, { "86210", "10.0.1.7:11211" },
	      { "86443", "10.0.1.4:11211" }, { "86676", "10.0.1.3:11211" },
	      { "86909", "10.0.1.1:11211" }, { "87142", "10.0.1.8:11211" },
	      { "87375", "10.0.1.1:11211" }, { "87608", "10.0.1.7:11211" },
	      { "87841", "10.0.1.1:11211" }, { "88074", "10.0.1.1:11211" },
	      { "88307", "10.0.1.7:11211" }, { "88540", "10.0.1.4:11211" },
	      { "88773", "10.0.1.5:11211" }, { "89006", "10.0.1.2:11211" },
	      { "89239", "10.0.1.1:11211" }, { "89472", "10.0.1.6:11211" },
	      { "89705", "10.0.1.2:11211" }, { "89938", "10.0.1.2:11211" },
	      { "90171", "10.0.1.7:11211" }, { "90404", "10.0.1.5:11211" },
	      { "90637", "10.0.1.8:11211" }, { "90870", "10.0.1.8:11211" },
	      { "91103", "10.0.1.7:11211" }, { "91336", "10.0.1.5:11211" },
	      { "91569", "10.0.1.8:11211" }, { "91802", "10.0.1.2:11211" },
	      { "92035", "10.0.1.8:11211" }, { "92268", "10.0.1.4:11211" },
	      { "92501", "10.0.1.6:11211" }, { "92734", "10.0.1.2:11211" },
	      { "92967", "10.0.1.6:11211" }, { "93200", "10.0.1.1:11211" },
	      { "93433", "10.0.1.2:11211" }, { "93666", "10.0.1.6:11211" },
	      { "93899", "10.0.1.2:11211" }, { "94132", "10.0.1.2:11211" },
	      { "103685", "10.0.1.2:11211" }, { "103918", "10.0.1.7:11211" },
	      { "104151", "10.0.1.5:11211" }, { "104384", "10.0.1.1:11211" },
	      { "104617", "10.0.1.3:11211" }, { "104850", "10.0.1.3:11211" },
	      { "105083", "10.0.1.7:11211" }, { "105316", "10.0.1.2:11211" },
	      { "105549", "10.0.1.3:11211" }, { "105782", "10.0.1.2:11211" },
	      { "106015", "10.0.1.4:11211" }, { "106248", "10.0.1.3:11211" },
	      { "106481", "10.0.1.4:11211" }, { "106714", "10.0.1.8:11211" },
	      { "106947", "10.0.1.8:11211" }, { "107180", "10.0.1.6:11211" },
	      { "107413", "10.0.1.7:11211" }, { "107646", "10.0.1.5:11211" },
	      { "107879", "10.0.1.8:11211" }, { "108112", "10.0.1.1:11211" },
	      { "108345", "10.0.1.8:11211" }, { "108578", "10.0.1.5:11211" },
	      { "112539", "10.0.1.8:11211" }, { "112772", "10.0.1.1:11211" },
	      { "113005", "10.0.1.5:11211" }, { "113238", "10.0.1.4:11211" },
	      { "116034", "10.0.1.8:11211" }, { "116267", "10.0.1.1:11211" },
	      { "116500", "10.0.1.6:11211" }, { "116733", "10.0.1.1:11211" },
	      { "116966", "10.0.1.6:11211" }, { "117199", "10.0.1.4:11211" },
	      { "117432", "10.0.1.1:11211" }, { "117665", "10.0.1.2:11211" },
	      { "117898", "10.0.1.6:11211" }, { "118131", "10.0.1.3:11211" },
	      { "118364", "10.0.1.2:11211" }, { "118597", "10.0.1.5:11211" },
	      { "118830", "10.0.1.5:11211" }, { "119063", "10.0.1.3:11211" },
	      { "119296", "10.0.1.6:11211" }, { "119529", "10.0.1.1:11211" },
	      { "119762", "10.0.1.6:11211" }, { "119995", "10.0.1.7:11211" },
	      { "120228", "10.0.1.2:11211" }, { "120461", "10.0.1.2:11211" },
	      { "124888", "10.0.1.3:11211" }, { "125121", "10.0.1.6:11211" },
	      { "125354", "10.0.1.5:11211" }, { "125587", "10.0.1.2:11211" },
	      { "125820", "10.0.1.3:11211" }, { "126053", "10.0.1.5:11211" },
	      { "126286", "10.0.1.5:11211" }, { "126519", "10.0.1.2:11211" },
	      { "126752", "10.0.1.6:11211" }, { "126985", "10.0.1.7:11211" },
	      { "127218", "10.0.1.6:11211" }, { "127451", "10.0.1.7:11211" },
	      { "127684", "10.0.1.6:11211" }, { "127917", "10.0.1.7:11211" },
	      { "128150", "10.0.1.6:11211" }, { "128383", "10.0.1.1:11211" },
	      { "128616", "10.0.1.4:11211" }, { "128849", "10.0.1.3:11211" },
	      { "129082", "10.0.1.5:11211" }, { "129315", "10.0.1.8:11211" },
	      { "129548", "10.0.1.6:11211" }, { "129781", "10.0.1.6:11211" },
	      { "130014", "10.0.1.6:11211" }, { "130247", "10.0.1.5:11211" },
	      { "130480", "10.0.1.6:11211" }, { "130713", "10.0.1.2:11211" },
	      { "130946", "10.0.1.5:11211" }, { "131179", "10.0.1.5:11211" },
	      { "131412", "10.0.1.7:11211" }, { "131645", "10.0.1.2:11211" },
	      { "131878", "10.0.1.6:11211" }, { "132111", "10.0.1.5:11211" },
	      { "132344", "10.0.1.8:11211" }, { "132577", "10.0.1.1:11211" },
	      { "132810", "10.0.1.1:11211" }, { "133043", "10.0.1.7:11211" },
	      { "133276", "10.0.1.4:11211" }, { "133509", "10.0.1.8:11211" },
	      { "133742", "10.0.1.3:11211" }, { "133975", "10.0.1.5:11211" },
	      { "134208", "10.0.1.1:11211" }, { "134441", "10.0.1.8:11211" },
	      { "134674", "10.0.1.7:11211" }, { "134907", "10.0.1.4:11211" },
	      { "135140", "10.0.1.3:11211" }, { "135373", "10.0.1.5:11211" },
	      { "135606", "10.0.1.7:11211" }, { "135839", "10.0.1.8:11211" },
	      { "136072", "10.0.1.8:11211" }, { "136305", "10.0.1.7:11211" },
	      { "136538", "10.0.1.1:11211" }, { "136771", "10.0.1.1:11211" },
	      { "137004", "10.0.1.2:11211" }, { "137237", "10.0.1.2:11211" } };

	    for (String[] s : exp) {
	      String k = s[0];
	      String server = s[1];
	      MemcachedNode n = locator.getPrimary(k);
	      assertEquals("/" + server, n.getSocketAddress().toString());
	    }

	  }

}
